// +build test

// That's right, we're testing the test code. Get on my level.

package model

import (
	"reflect"
	"testing"

	"ting/util"
)

func TestFactoryAttr(t *testing.T) {
	t.Run("With", attrWithTests)
	t.Run("Get", attrGetTests)
	t.Run("Getp", attrGetpTests)
}

func attrWithTests(t *testing.T) {
	t.Parallel()

	attr := Attr{"original": 10, "replaced": 20}
	attr2 := attr.With("replaced", 200).With("new", 30)

	attr.AssertEqual(t, Attr{"original": 10, "replaced": 20})
	attr2.AssertEqual(t, Attr{"original": 10, "replaced": 200, "new": 30})
}

func attrGetTests(t *testing.T) {
	t.Run("NilAttr", func(t *testing.T) {
		var attr Attr = nil

		t.Run("Default", func(t *testing.T) {
			t.Parallel()
			result := attr.Get("foo", "bar")
			util.AssertEqual(t, result, "bar")
		})

		t.Run("NoDefault", func(t *testing.T) {
			t.Parallel()
			// Pretty much just checking for no panic here.
			if result := attr.Get("foo", nil); result != nil {
				t.Fatalf("unexpected result: %#v; expected nil", result)
			}
		})
	})

	t.Run("Found", func(t *testing.T) {
		t.Parallel()
		result := (Attr{"foo": "bar"}).Get("foo", "baz")
		util.AssertEqual(t, result, "bar")
	})

	t.Run("DefaultNil", func(t *testing.T) {
		t.Parallel()
		if result := (Attr{}).Get("foo", nil); result != nil {
			t.Fatalf("unexpected result: %#v; expected nil", result)
		}
	})

	t.Run("DefaultValue", func(t *testing.T) {
		t.Parallel()
		result := (Attr{}).Get("foo", "bar")
		util.AssertEqual(t, result, "bar")
	})

	t.Run("DefaultFunction", func(t *testing.T) {
		t.Parallel()
		result := (Attr{}).Get("foo", func() string { return "bar" })
		util.AssertEqual(t, result, "bar")
	})
}

func attrGetpTests(t *testing.T) {
	t.Run("NilAttr", func(t *testing.T) {
		var attr Attr = nil

		t.Run("Default", func(t *testing.T) {
			t.Parallel()
			result := attr.Getp("foo", "bar")
			if _, ok := result.(*string); !ok {
				t.Fatalf("wrong type: %T; expected *string", result)
			}
			util.AssertEqual(t, result, "bar")
		})

		t.Run("NoDefault", func(t *testing.T) {
			t.Parallel()
			if result := attr.Getp("foo", nil); result != nil {
				t.Fatalf("unexpected result: %#v; expected nil", result)
			}
		})
	})

	strPtr := reflect.PtrTo(reflect.TypeOf(""))
	assertStrPtr := func(t *testing.T, v interface{}) {
		t.Helper()
		if v == nil || reflect.TypeOf(v) != strPtr {
			t.Fatalf("wrong type: %T; expected *string", v)
		}
	}

	t.Run("Found", func(t *testing.T) {
		t.Parallel()
		result := (Attr{"foo": "bar"}).Getp("foo", "baz")
		assertStrPtr(t, result)
		util.AssertEqual(t, result, "bar")
	})

	t.Run("DefaultNil", func(t *testing.T) {
		t.Parallel()
		if result := (Attr{}).Getp("foo", nil); result != nil {
			t.Fatalf("unexpected result: %#v; expected nil", result)
		}
	})

	t.Run("DefaultValue", func(t *testing.T) {
		t.Parallel()
		result := (Attr{}).Getp("foo", "bar")
		assertStrPtr(t, result)
		util.AssertEqual(t, result, "bar")
	})

	t.Run("DefaultPtrValue", func(t *testing.T) {
		t.Parallel()
		exp := "bar"
		result := (Attr{}).Getp("foo", &exp)
		assertStrPtr(t, result)
		util.AssertEqual(t, result, exp)
	})

	t.Run("DefaultFunction", func(t *testing.T) {
		t.Parallel()
		result := (Attr{}).Getp("foo", func() string { return "bar" })
		assertStrPtr(t, result)
		util.AssertEqual(t, result, "bar")
	})

	t.Run("DefaultPtrFunction", func(t *testing.T) {
		t.Parallel()
		exp := "bar"
		result := (Attr{}).Getp("foo", func() *string { return &exp })
		assertStrPtr(t, result)
		util.AssertEqual(t, result, "bar")
	})
}
