package model

import (
	"database/sql"
	"fmt"
)

const (
	LeaderSortCorrect = "correct"
	LeaderSortPercent = "percent"
)

type Leaderboard struct {
	ChannelID int                 `json:"channel_id"`
	Sort      string              `json:"sort"`
	Limit     int                 `json:"limit"`
	Entries   []*LeaderboardEntry `json:"entries"`
}

type LeaderboardEntry struct {
	UserID   int     `db:"user_id"   json:"user_id"`
	User     *User   `db:"-"         json:"user"`
	Correct  int     `db:"correct"   json:"correct"`
	Answered int     `db:"answered"  json:"answered"`
	Percent  float64 `db:"percent"   json:"percent"`
}

func orderStmt(sort string) (string, error) {
	switch sort {
	case LeaderSortCorrect:
		return `ORDER BY "correct" DESC`, nil
	case LeaderSortPercent:
		return `ORDER BY "percent" DESC`, nil
	default:
		return "", fmt.Errorf("invalid leaderboard sort: %q", sort)
	}
}

func (db *DB) GetLeaderboard(channelID int, sort string, limit int) (*Leaderboard, Error) {
	orderSQL, err := orderStmt(sort)
	if err != nil {
		return nil, UserError(err)
	}

	leaderboard := Leaderboard{
		ChannelID: channelID,
		Sort:      sort,
		Limit:     limit,
		Entries:   []*LeaderboardEntry{},
	}

	if _, chErr := db.FindOrCreateChannel(channelID); chErr != nil {
		return nil, chErr.Prefix("error looking up channel %d", channelID)
	}

	if err := db.Select(&leaderboard.Entries,
		`SELECT "user_id", "correct", "answered", "percent" FROM "leaderboard" WHERE "channel_id" = $1 `+orderSQL+` LIMIT $2`,
		channelID, limit,
	); err != nil && err != sql.ErrNoRows {
		return nil, DBError(err)
	}

	for _, entry := range leaderboard.Entries {
		var err Error
		if entry.User, err = db.FindUser(entry.UserID); err != nil {
			return nil, err.Prefix("error fetching details for user %d", entry.UserID)
		} else if entry.User == nil {
			return nil, DBErrorf("leaderboard includes unrecognized user: %d", entry.UserID)
		}
	}

	return &leaderboard, nil
}
