// +build test

package model

import (
	"fmt"
	"testing"

	uuid "github.com/satori/go.uuid"
)

func TestLeaderboard(t *testing.T) {
	db := InitTestDB(t)
	defer db.Close()

	t.Run("GetLeaderboard", getLeaderboardTests(db))
}

func getLeaderboardTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		t.Run("NoAnswers", func(t *testing.T) {
			channel := db.NewTestChannel(t, nil)
			if lb, err := db.GetLeaderboard(channel.ID, LeaderSortCorrect, 5); err != nil {
				t.Fatalf("unexpected error: %s", err)
			} else if lb.ChannelID != channel.ID {
				t.Fatalf("result has wrong channel ID: %d; expected: %d", lb.ChannelID, channel.ID)
			} else if lb.Entries == nil {
				t.Fatal("unexpected nil for entries; expected non-nil but empty")
			}
		})

		t.Run("NoneCorrect", func(t *testing.T) {
			ansAttrs := Attr{"IsCorrect": false}
			q := db.NewTestQuestion(t, Attr{"Answers": []Attr{ansAttrs, ansAttrs, ansAttrs}})
			if q.HasCorrect() {
				t.Fatalf("question not expected to have a correct answer: %#v", q.Answers)
			} else if _, err := q.AddUserAnswer(db.NewTestUser(t, nil).OpaqueID, q.AnswerIDs[0], true); err != nil {
				t.Fatalf("unexpected error adding answer: %s", err)
			} else if lb, err := db.GetLeaderboard(q.ChannelID, LeaderSortCorrect, 5); err != nil {
				t.Fatalf("unexpected error getting leaderboard: %s", err)
			} else if len(lb.Entries) > 0 {
				t.Fatalf("unexpected leaderboard entries: %#v", lb.Entries)
			}
		})

		t.Run("Sort", func(t *testing.T) {
			channel := db.NewTestChannel(t, nil)

			// 5 Users:
			// [0] = 1/1 correct (100%)
			// [1] = 2/3 correct  (66%)
			// [2] = 3/5 correct  (60%)
			// [3] = 4/7 correct (~57%)
			// [4] = 5/9 correct (~56%)
			// Note that sort by #correct is reverse of sort by %correct.
			baseUserID := uuid.NewV4().String()
			userID := func(i int) string { return fmt.Sprintf("%s:%d", baseUserID, i) }
			users := make([]*User, 5)
			for i := 0; i < 5; i++ {
				users[i] = db.NewTestUser(t, Attr{"OpaqueID": userID(i)})
			}
			shuffleUsers(users) // To avoid accidental test pass if ordered by ID.

			// 9 questions, each with two answers: the first correct, the second incorrect.
			questions := make([]*Question, 9)
			for i := 0; i < 9; i++ {
				questions[i] = db.NewTestQuestion(t, Attr{
					"ChannelID": channel.ID,
					"Answers":   []Attr{Attr{"IsCorrect": true}, Attr{"IsCorrect": false}},
				})
			}

			answerIndex := map[bool]int{true: 0, false: 1} // ternary expression, lul
			for iUser, user := range users {
				maxQuestions := 1 + 2*iUser
				maxCorrect := 1 + iUser
				for i := 1; i <= maxQuestions; i++ {
					q := questions[i-1]
					answerID := q.AnswerIDs[answerIndex[i <= maxCorrect]]
					if _, err := q.AddUserAnswer(user.OpaqueID, answerID, true); err != nil {
						t.Fatalf("unexpected error adding user %d answer %d/%d to question %d: %s",
							user.ID, i, maxQuestions, q.ID, err.Error())
					}
				}
			}

			var lb *Leaderboard
			var order string
			var err Error

			// Extra context for test failures:
			defer func() {
				if !t.Failed() || lb == nil {
					return
				}
				t.Logf("users (correct ASC / percent DESC):")
				for i, user := range users {
					t.Logf("  %d. User %d (opaque %q)", i+1, user.ID, user.OpaqueID)
				}
				t.Logf("entries:")
				for i, entry := range lb.Entries {
					t.Logf("  %d. User %d (opaque %q): %d/%d correct (%.2f%%)",
						i+1, entry.UserID, entry.User.OpaqueID, entry.Correct, entry.Answered, 100*entry.Percent)
				}
			}()

			for _, order = range []string{LeaderSortCorrect, LeaderSortPercent} {
				lb, err = db.GetLeaderboard(channel.ID, order, 10)
				if err != nil {
					t.Fatalf("unexpected error getting leaderboard by %q for channel %d: %s", order, channel.ID, err)
				} else if len(lb.Entries) != 5 {
					t.Fatalf("wrong number of leaderboard entries: %d; expected: %d", len(lb.Entries), 5)
				}

				for iEntry, entry := range lb.Entries {
					entryStr := fmtPtr(entry)
					iUser := iEntry
					if order == LeaderSortCorrect {
						iUser = 4 - iEntry
					}
					answered := 1 + 2*iUser
					correct := 1 + iUser
					expEntryStr := fmtPtr(&LeaderboardEntry{
						UserID:   users[iUser].ID,
						User:     users[iUser],
						Correct:  correct,
						Answered: answered,
						Percent:  float64(correct) / float64(answered),
					})
					if entryStr != expEntryStr {
						t.Fatalf("wrong entry at leaderboard (%q) position %d/%d:\nfound: %s\nexpected: %s",
							order, iEntry+1, len(lb.Entries), entryStr, expEntryStr)
					}
				}
			}
		})
	}
}
