package model

import (
	"bytes"
	"fmt"
	"strings"
)

type sqlSelecter interface {
	Select(dest interface{}, query string, args ...interface{}) error
}

type queryBuilder struct {
	table   string
	cols    []string
	colsRaw []bool
	vars    []string
	args    []interface{}

	whereCols []string
	whereArgs []interface{}
}

func newQueryBuilder(table string) *queryBuilder {
	return &queryBuilder{
		table:     table,
		cols:      nil,
		colsRaw:   nil,
		vars:      nil,
		args:      nil,
		whereCols: nil,
		whereArgs: nil,
	}
}

func (qb *queryBuilder) add(col string, raw bool, varAnnot string, val interface{}) {
	qb.cols = append(qb.cols, col)
	qb.colsRaw = append(qb.colsRaw, raw)
	qb.vars = append(qb.vars, fmt.Sprintf("$%d%s", len(qb.vars)+1, varAnnot))
	qb.args = append(qb.args, val)
}

func (qb *queryBuilder) Add(col string, val interface{}) {
	qb.add(col, false, "", val)
}

func (qb *queryBuilder) AddAnnot(col string, varAnnot string, val interface{}) {
	qb.add(col, false, varAnnot, val)
}

func (qb *queryBuilder) AddRaw(col string, val interface{}) {
	qb.add(col, true, "", val)
}

func (qb *queryBuilder) AddRawAnnot(col string, varAnnot string, val interface{}) {
	qb.add(col, true, varAnnot, val)
}

func (qb *queryBuilder) AddWhere(col string, val interface{}) {
	qb.whereCols = append(qb.whereCols, col)
	qb.whereArgs = append(qb.whereArgs, val)
}

func (qb *queryBuilder) Len() int {
	return len(qb.cols)
}

func (qb *queryBuilder) Cols() string {
	if len(qb.cols) == 1 {
		return fmt.Sprintf("%q", qb.cols[0])
	}
	buf := bytes.NewBufferString(fmt.Sprintf("(%q", qb.cols[0]))
	for i := 1; i < len(qb.cols); i++ {
		buf.WriteString(fmt.Sprintf(", %q", qb.cols[i]))
	}
	buf.WriteString(")")
	return buf.String()
}

func (qb *queryBuilder) Vars() string {
	if len(qb.vars) == 1 {
		return qb.vars[0]
	}
	return "(" + strings.Join(qb.vars, ", ") + ")"
}

func (qb *queryBuilder) Where() string {
	if len(qb.whereCols) == 0 {
		return ""
	}
	clauses := make([]string, len(qb.whereCols))
	offset := len(qb.vars) + 1
	for i, col := range qb.whereCols {
		clauses[i] = fmt.Sprintf("%q = $%d", col, offset+i)
	}
	return " WHERE " + strings.Join(clauses, " AND ")
}

func (qb *queryBuilder) Args() []interface{} {
	return append(qb.args, qb.whereArgs...)
}

func (qb *queryBuilder) Update() string {
	return fmt.Sprintf(`UPDATE %q SET %s = %s%s`, qb.table, qb.Cols(), qb.Vars(), qb.Where())
}
