package model

import (
	"encoding/json"
	"time"

	"ting/util"
	"ting/util/set"

	"github.com/jmoiron/sqlx"
)

func jsonInt(v interface{}) (int, bool) {
	if x, ok := v.(int); ok {
		return x, true
	} else if xF, ok := v.(float64); !ok {
		return 0, false
	} else if x = int(xF); float64(x) != xF {
		return 0, false
	} else {
		return x, true
	}
}

func (qOrig *Question) Update(updates map[string]interface{}) (*Question, Error) {
	if now := time.Now(); qOrig.ActiveFrom != nil && qOrig.ActiveFrom.Before(now) {
		status := "active"
		if qOrig.ActiveUntil.Before(now) {
			status = "expired"
		}
		return nil, UserErrorf("cannot modify %s question", status)
	}

	tx, txErr := qOrig.db.Beginx()
	if txErr != nil {
		return nil, DBErrorf("error initializing transaction: %s", txErr)
	}
	defer tx.Rollback()

	if qParams, err := questionUpdateQB(tx, qOrig, updates); err != nil {
		return nil, err
	} else if qParams.Len() > 0 {
		qParams.AddWhere("id", qOrig.ID)
		if _, err := tx.Exec(qParams.Update(), qParams.Args()...); err != nil {
			return nil, DBErrorf("error updating question: %s", err.Error())
		}
	}

	if ansUpdatesV, found := updates["answers"]; found {
		if ansUpdatesV == nil {
			return nil, UserErrorf("answers: must have at least one")
		}
		ansUpdates, ok := ansUpdatesV.([]interface{})
		if !ok {
			return nil, UserErrorf("answers: must be an array; found %T", ansUpdatesV)
		} else if len(ansUpdates) == 0 {
			return nil, UserErrorf("answers: must have at least one")
		}

		unusedIDs := set.NewIntSet(qOrig.AnswerIDs...)

		for i, ansUpdateV := range ansUpdates {
			if ansUpdateV == nil {
				return nil, UserErrorf("answer %d: cannot be null", i+1)
			}
			ansUpdate, ok := ansUpdateV.(map[string]interface{})
			if !ok {
				return nil, UserErrorf("answer %d: must be an object", i+1)
			} else if idV, found := ansUpdate["id"]; found && idV != nil {
				// Request had an Answer ID -- look it up and modify it in place.
				id, ok := jsonInt(idV)
				if !ok {
					return nil, UserErrorf("answer %d: id must be an integer", i+1)
				}
				unusedIDs.Remove(id)

				if aOrig := qOrig.FindAnswer(id); aOrig == nil {
					return nil, UserErrorf("answer %d: question has no answer with this ID", i+1)
				} else if aParams, err := answerUpdatesQB(aOrig, ansUpdate); err != nil {
					return nil, UserErrorf("answer %d: %s", i+1, err.Error())
				} else if aParams.Len() > 0 {
					aParams.AddWhere("id", id)
					if _, err := tx.Exec(aParams.Update(), aParams.Args()...); err != nil {
						return nil, DBErrorf("error updating answer %d (id=%d): %s", i+1, id, err)
					}
				}
			} else {
				// No Answer ID -- create new question.
				var answer NewAnswer
				if buf, err := json.Marshal(ansUpdate); err != nil {
					return nil, DBErrorf("answer %d: error re-marshaling object JSON: %s", i+1, err)
				} else if err := json.Unmarshal(buf, &answer); err != nil {
					return nil, UserErrorf("answer %d: JSON parse error", i+1)
				} else if dbErr := createAnswer(tx, qOrig.ID, answer); dbErr != nil {
					return nil, DBErrorf("answer %d: %s", i+1, dbErr.Error())
				}
			}
		}

		// Delete any answers that weren't included in the request.
		for _, id := range unusedIDs.Values() {
			if _, err := tx.Exec(`DELETE FROM "answers" WHERE "id" = $1`, id); err != nil {
				return nil, DBErrorf("error deleting answer %d: %s", id, err)
			}
		}
	}

	if err := tx.Commit(); err != nil {
		return nil, DBErrorf("error committing transaction: %s", err)
	}

	return qOrig.db.FindQuestion(qOrig.ID)
}

func questionUpdateQB(tx *sqlx.Tx, qOrig *Question, updates map[string]interface{}) (*queryBuilder, Error) {
	ignoredFields := []string{"responses", "answer_ids", "answered", "user_answer"}
	for _, field := range ignoredFields {
		delete(updates, field)
	}

	staticFields := map[string]int{
		"id":         qOrig.ID,
		"channel_id": qOrig.ChannelID,
	}
	for field, exp := range staticFields {
		if v, found := updates[field]; found {
			if x, ok := jsonInt(v); !ok {
				return nil, UserErrorf("%s: must be an integer", field, v)
			} else if x != exp {
				return nil, UserErrorf("%s: cannot be modified", field)
			}
			delete(updates, field)
		}
	}

	foundKeys := set.FromStringMap(updates)
	goodKeys := set.NewStringSet("text", "active_from", "active_until", "settings", "answers")
	if badKeys := foundKeys.Diff(goodKeys).Values(); len(badKeys) > 0 {
		return nil, UserErrorf("unrecognized fields: %s", util.FmtStrings(badKeys))
	}

	qb := newQueryBuilder("questions")

	if textV, found := updates["text"]; found {
		if text, ok := textV.(string); !ok {
			return nil, UserErrorf("text: not a string")
		} else if len(text) == 0 {
			return nil, UserErrorf("text: cannot be blank")
		} else if text != qOrig.Text {
			qb.Add("text", text)
		}
	}

	newActive := false
	var from, until *time.Time
	var timeErr error
	if fromV, found := updates["active_from"]; found {
		if from, timeErr = util.ParseTimep(fromV); timeErr != nil {
			return nil, UserErrorf("active_from: %s", timeErr)
		} else if util.XorNil(from, qOrig.ActiveFrom) || (from != nil && !from.Equal(*qOrig.ActiveFrom)) {
			newActive = true
			qb.Add("active_from", from)
		}
	}
	if untilV, found := updates["active_until"]; found {
		if until, timeErr = util.ParseTimep(untilV); timeErr != nil {
			return nil, UserErrorf("active_until: %s", timeErr)
		} else if util.XorNil(until, qOrig.ActiveUntil) || (until != nil && !until.Equal(*qOrig.ActiveUntil)) {
			newActive = true
			qb.Add("active_until", until)
		}
	}
	if util.XorNil(from, until) {
		return nil, UserErrorf("active_from and active_until must either be both nil or both non-nil")
	} else if newActive && from != nil && !from.Before(*until) {
		return nil, UserErrorf("active_from must be before active_until")
	} else if newActive && from != nil {
		if ids, err := overlappingQuestionIDs(tx, qOrig.ChannelID, *from, *until); err != nil {
			return nil, DBErrorf("error checking conflicts for new active time: %s", err)
		} else if !(len(ids) == 0 || (len(ids) == 1 && ids[0] == qOrig.ID)) {
			return nil, UserErrorf("question overlaps existing questions").WithMeta("conflicts", ids)
		}
	}

	if settingsV, found := updates["settings"]; found {
		settingsMap, ok := settingsV.(map[string]interface{})
		if !ok {
			return nil, UserErrorf("settings: expected object")
		}

		foundKeys := set.FromStringMap(settingsMap)
		goodKeys := set.NewStringSet("enable_chat")
		if badKeys := foundKeys.Diff(goodKeys).Values(); len(badKeys) > 0 {
			return nil, UserErrorf("settings: unrecognized settings: %s", util.FmtStrings(badKeys))
		}

		changed := false

		if chatV, found := settingsMap["enable_chat"]; found {
			if chat, ok := chatV.(bool); !ok {
				return nil, UserErrorf("settings.enable_chat: not a boolean")
			} else if chat != qOrig.Settings.EnableChat {
				changed = true
			}
		}

		if changed {
			if settingsJSON, err := json.Marshal(settingsMap); err != nil {
				return nil, UserErrorf("settings: error marshaling back to JSON: %s", err)
			} else {
				qb.AddAnnot("settings", "::jsonb", string(settingsJSON))
			}
		}
	}

	return qb, nil
}

func answerUpdatesQB(aOrig *Answer, updates map[string]interface{}) (*queryBuilder, Error) {
	ignore := []string{"question", "responses"}
	for _, field := range ignore {
		delete(updates, field)
	}

	static := map[string]int{"id": aOrig.ID, "question_id": aOrig.QuestionID}
	for field, exp := range static {
		if idV, found := updates[field]; found {
			if id, ok := jsonInt(idV); !ok {
				return nil, UserErrorf("%s: must be an integer", field)
			} else if id != exp {
				return nil, UserErrorf("%s: cannot be modified", field)
			}
			delete(updates, field)
		}
	}

	foundKeys := set.FromStringMap(updates)
	goodKeys := set.NewStringSet("text", "emote_id", "match", "is_correct")
	if badKeys := foundKeys.Diff(goodKeys).Values(); len(badKeys) > 0 {
		return nil, UserErrorf("unrecognized fields: %s", util.FmtStrings(badKeys))
	}

	qb := newQueryBuilder("answers")

	if textV, found := updates["text"]; found {
		if text, ok := textV.(string); !ok {
			return nil, UserErrorf("text: not a string")
		} else if len(text) == 0 {
			return nil, UserErrorf("text: cannot be blank")
		} else if text != aOrig.Text {
			qb.Add("text", text)
		}
	}

	if emoteV, found := updates["emote_id"]; found {
		if emoteV == nil {
			if aOrig.EmoteID != nil {
				qb.Add("emote_id", nil)
			}
		} else if emoteID, ok := jsonInt(emoteV); !ok {
			return nil, UserErrorf("emote_id: must be an integer")
		} else if aOrig.EmoteID == nil || emoteID != *aOrig.EmoteID {
			qb.Add("emote_id", emoteID)
		}
	}

	if matchV, found := updates["match"]; found {
		if matchV == nil {
			if aOrig.Match != nil {
				qb.Add("match", nil)
			}
		} else if match, ok := matchV.(string); !ok {
			return nil, UserErrorf("match: must be a string")
		} else if aOrig.Match == nil || match != *aOrig.Match {
			qb.Add("match", match)
		}
	}

	if correctV, found := updates["is_correct"]; found {
		if correct, ok := correctV.(bool); !ok {
			return nil, UserErrorf("is_correct: must be a boolean")
		} else if correct != aOrig.IsCorrect {
			qb.Add("is_correct", correct)
		}
	}

	return qb, nil
}
