// +build test

package model

import (
	"bytes"
	"encoding/json"
	"fmt"
	"math/rand"
	"sort"
	"strconv"
	"strings"
	"testing"
	"time"

	"ting/util"
)

// This function kinda snowballed... the last cases should probably be `String()` methods or something. >_>
func fmtPtr(v interface{}) string {
	if util.IsNil(v) {
		return "(nil)"
	}
	switch x := v.(type) {
	case *int:
		return strconv.Itoa(*x)
	case *string:
		return fmt.Sprintf("%q", *x)
	case *bool:
		return fmt.Sprint(*x)
	case *time.Time:
		return fmt.Sprintf("%q", x.Format(time.RFC3339))
	case []int:
		return fmt.Sprint(x)
	case *Answer:
		return fmt.Sprintf("Answer(id=%d, qid=%d, %q, emote=%s, match=%s, correct=%v, resp=%d)",
			x.ID, x.QuestionID, x.Text, fmtPtr(x.EmoteID), fmtPtr(x.Match), x.IsCorrect, x.Responses)
	case *UserAnswer:
		return fmt.Sprintf("UserAnswer(u=%d, q=%d, a=%s, msg=%s)",
			x.UserID, x.QuestionID, fmtPtr(x.AnswerID), fmtPtr(x.Message))
	case *LeaderboardEntry:
		return fmt.Sprintf("User %d (opaque %q): %d/%d correct (%.2f%%)",
			x.UserID, x.User.OpaqueID, x.Correct, x.Answered, x.Percent*100.0)
	default:
		return fmt.Sprintf("%#v", v)
	}
}

func makeIndent(level int) string {
	indent := ""
	for i := 0; i < level; i++ {
		indent += "    "
	}
	return indent
}

func (m1 Attr) AssertEqual(t *testing.T, m2 Attr) {
	t.Helper()
	if util.XorNil(m1, m2) {
		t.Fatalf("one Attr is nil and the other is not: %#v vs %#v", m1, m2)
	} else if m1 == nil && m2 == nil {
		return
	} else if len(m1) != len(m2) {
		t.Fatalf("Attr lengths differ: %d vs %d", len(m1), len(m2))
	}
	for k, v1 := range m1 {
		// I miss the `Eq` typeclass.
		if v2, found := m2[k]; !found {
			t.Errorf("key %q not found in right Attr; value in left Attr: %#v", k, v1)
		} else if v1Str, v2Str := fmtPtr(v1), fmtPtr(v2); v1Str != v2Str {
			t.Errorf("Attr values differ for key %q: %s vs %s", k, v1Str, v1Str, v2Str)
		}
	}
	if t.Failed() {
		t.FailNow()
	}
}

func (c1 *Channel) AssertEqual(t *testing.T, c2 *Channel) {
	t.Helper()
	if diffs := c1.diff(c2, 1); len(diffs) > 0 {
		t.Fatalf("unexpected differences in channels (%d):\n%s", len(diffs), strings.Join(diffs, "\n"))
	}
	util.AssertEqual(t, c1.Settings.ThemeSettings, c2.Settings.ThemeSettings)
}

func (c1 *Channel) diff(c2 *Channel, indentLvl int) (diffs []string) {
	indent := makeIndent(indentLvl)
	if c1.ID != c2.ID {
		diffs = append(diffs, indent+fmt.Sprintf("ID: %d vs %d", c1.ID, c2.ID))
	}
	if c1.Settings.Theme != c2.Settings.Theme {
		diffs = append(diffs, indent+fmt.Sprintf("Theme: %q vs %q", c1.Settings.Theme, c2.Settings.Theme))
	}
	if id1, id2 := c1.CurrentQuestionID, c2.CurrentQuestionID; util.XorNil(id1, id2) || (id1 != nil && *id1 != *id2) {
		diffs = append(diffs, indent+fmt.Sprintf("CurrentQuestionID: %s vs %s", fmtPtr(id1), fmtPtr(id2)))
	}
	if q1, q2 := c1.CurrentQuestion, c2.CurrentQuestion; util.XorNil(q1, q2) {
		diffs = append(diffs, indent+fmt.Sprintf("CurrentQuestion: %#v vs %#v", q1, q2))
	} else if q1 != nil && q2 != nil {
		if qDiffs := q1.diff(q2, indentLvl+1); len(qDiffs) > 0 {
			diffs = append(diffs, indent+"CurrentQuestion:\n"+strings.Join(qDiffs, ""))
		}
	}
	return diffs
}

func (c *Channel) AssertCurrentQuestion(t *testing.T, q *Question) {
	t.Helper()
	if q == nil {
		if c.CurrentQuestionID != nil {
			t.Fatalf("unexpected current question ID: %d", *c.CurrentQuestionID)
		} else if c.CurrentQuestion != nil {
			t.Fatalf("unexpected current question: %#v", *c.CurrentQuestion)
		}
		return
	}

	if c.CurrentQuestionID == nil {
		t.Fatalf("expected non-nil current question ID (%d)", q.ID)
	} else if *c.CurrentQuestionID != q.ID {
		t.Fatalf("wrong current question ID: %d; expected %d", c.CurrentQuestionID, q.ID)
	} else if c.CurrentQuestion == nil {
		t.Fatalf("expected non-nil current question (ID %d)", q.ID)
	} else {
		c.CurrentQuestion.AssertEqual(t, q)
	}
}

func (q1 *Question) AssertEqual(t *testing.T, q2 *Question) {
	t.Helper()
	if diffs := q1.diff(q2, 1); len(diffs) > 0 {
		t.Fatalf("unexpected differences in questions (%d):\n%s", len(diffs), strings.Join(diffs, "\n"))
	}
}

func (q1 *Question) diff(q2 *Question, indentLvl int) (diffs []string) {
	indent := makeIndent(indentLvl)

	for _, field := range []struct {
		name   string
		v1, v2 interface{}
	}{
		{name: "ID", v1: &q1.ID, v2: &q2.ID},
		{name: "ChannelID", v1: &q1.ChannelID, v2: &q2.ChannelID},
		{name: "Text", v1: &q1.Text, v2: &q2.Text},
		{name: "ActiveFrom", v1: q1.ActiveFrom, v2: q2.ActiveFrom},
		{name: "ActiveUntil", v1: q1.ActiveUntil, v2: q2.ActiveUntil},
		{name: "Responses", v1: &q1.Responses, v2: &q2.Responses},
		{name: "Answered", v1: q1.Answered, v2: q2.Answered},
		{name: "UserAnswer", v1: q1.UserAnswer, v2: q2.UserAnswer},
		{name: "AnswerIDs", v1: q1.AnswerIDs, v2: q2.AnswerIDs},
	} {
		if s1, s2 := fmtPtr(field.v1), fmtPtr(field.v2); s1 != s2 {
			diffs = append(diffs, indent+fmt.Sprintf("%s: %s vs %s", field.name, s1, s2))
		}
	}

	if sDiffs := util.StructDiff(q1.Settings, q2.Settings); len(sDiffs) > 0 {
		diffs = append(diffs, indent+"Settings: "+util.DiffErrorMessage(sDiffs, indentLvl+1))
	}

	if (q1.Answers == nil && q2.Answers != nil) || (q1.Answers != nil && q2.Answers == nil) || len(q1.Answers) != len(q2.Answers) {
		buf := bytes.NewBufferString(indent + "Answers:\n")
		fmtAnswers := func(arg string, ans []*Answer) {
			if len(ans) == 0 {
				buf.WriteString(indent + fmt.Sprintf("    %s: (none)\n", arg))
			} else {
				buf.WriteString(indent + fmt.Sprintf("    %s (%d):\n", arg, len(ans)))
				for _, a := range ans {
					buf.WriteString(indent + "        " + fmtPtr(a) + "\n")
				}
			}
		}
		fmtAnswers("left", q1.Answers)
		fmtAnswers("right", q2.Answers)
		diffs = append(diffs, buf.String())
	} else {
		for i := 0; i < len(q1.Answers); i++ {
			a1 := fmtPtr(q1.Answers[i])
			a2 := fmtPtr(q2.Answers[i])
			if a1 != a2 {
				buf := new(bytes.Buffer)
				buf.WriteString(indent + fmt.Sprintf("Answer %d/%d:\n", i+1, len(q1.Answers)))
				buf.WriteString(indent + fmt.Sprintf("    left: %s\n", a1))
				buf.WriteString(indent + fmt.Sprintf("    right: %s\n", a2))
				diffs = append(diffs, buf.String())
			}
		}
	}

	return diffs
}

func (q *Question) SimpleCheck(t *testing.T, text string, from, until *time.Time, answers []string, correct []int) {
	t.Helper()

	if q.Text != text {
		t.Errorf("wrong question text: %q; expected %q", q.Text, text)
	}
	if fromExp, fromAct := fmtPtr(from), fmtPtr(q.ActiveFrom); fromExp != fromAct {
		t.Errorf("wrong active_from: %q; expected %q", fromAct, fromExp)
	}
	if untilExp, untilAct := fmtPtr(until), fmtPtr(q.ActiveUntil); untilExp != untilAct {
		t.Errorf("wrong active_until: %q; expected %q", untilAct, untilExp)
	}
	for i, answer := range answers {
		expected := q.Answers[i].Text
		if answer != expected {
			t.Errorf("answer %d/%d has wrong text: %q; expected %q", i+1, len(answers), answer, expected)
		}
	}
	for _, i := range correct {
		if !q.Answers[i].IsCorrect {
			t.Errorf("expected Answers[%d] to be a correct answer", i)
		}
	}
	if t.Failed() {
		t.FailNow()
	}
}

func (q *Question) setActive(t *testing.T, active bool) {
	t.Helper()
	from, until, activeStr := "NULL", "NULL", "inactive"
	if active {
		from = `(NOW() AT TIME ZONE 'UTC' - INTERVAL '1 HOUR')`
		until = `(NOW() AT TIME ZONE 'UTC' + INTERVAL '1 HOUR')`
		activeStr = "active"
	}
	if err := q.db.Get(q,
		`UPDATE "questions" SET ("active_from", "active_until") = (`+from+`, `+until+`) WHERE "id" = $1 RETURNING *`,
		q.ID,
	); err != nil {
		t.Fatalf("unexpected error making question %d %s: %s", q.ID, activeStr, err)
	}
}

func questionIDs(qs []*Question) []int {
	ids := make([]int, len(qs))
	for i, q := range qs {
		ids[i] = q.ID
	}
	return ids
}

func questionTimes(dayDelta int) (from, until time.Time) {
	t := time.Now().Add(24 * time.Duration(dayDelta) * time.Hour).UTC()
	return t.Add(-1 * time.Hour), t.Add(1 * time.Hour)
}

func startOfDay(t time.Time) time.Time {
	y, m, d := t.Date()
	return time.Date(y, m, d, 0, 0, 0, 0, t.Location())
}

func endOfDay(t time.Time) time.Time {
	y, m, d := t.Date()
	return time.Date(y, m, d, 23, 59, 59, 999999, t.Location())
}

func (u1 *User) AssertEqual(t *testing.T, u2 *User) {
	t.Helper()
	if diffs := util.StructDiff(*u1, *u2); len(diffs) > 0 {
		t.Fatalf("unexpected differences in users: %s", util.DiffErrorMessage(diffs, 1))
	}
}

func (ua1 *UserAnswer) AssertEqual(t *testing.T, ua2 *UserAnswer) {
	t.Helper()
	if s1, s2 := fmtPtr(ua1), fmtPtr(ua2); s1 != s2 {
		t.Fatalf("UserAnswer mismatch:\nleft: %s\nright: %s", s1, s2)
	}
}

// Yay, no generics: support for shuffling an array of `*User`.
type shufUsers []*User

func (u shufUsers) Len() int           { return len(u) }
func (u shufUsers) Less(_, _ int) bool { return rand.Intn(2) == 0 }
func (u shufUsers) Swap(i, j int)      { u[i], u[j] = u[j], u[i] }
func shuffleUsers(users []*User)       { sort.Sort(shufUsers(users)) }

func parseObject(t *testing.T, obj interface{}, jsonFmt string, jsonArgs ...interface{}) {
	t.Helper()
	jsonBody := []byte(fmt.Sprintf(jsonFmt, jsonArgs...))
	if err := json.Unmarshal(jsonBody, obj); err != nil {
		t.Fatalf("error parsing JSON for %T: %s", obj, err)
	}
}

func ParseChannel(t *testing.T, jsonFmt string, jsonArgs ...interface{}) *Channel {
	t.Helper()
	var c Channel
	parseObject(t, &c, jsonFmt, jsonArgs...)
	return &c
}

func ParseQuestion(t *testing.T, jsonFmt string, jsonArgs ...interface{}) *Question {
	t.Helper()
	var q Question
	parseObject(t, &q, jsonFmt, jsonArgs...)
	return &q
}

func ParseQuestions(t *testing.T, jsonFmt string, jsonArgs ...interface{}) []*Question {
	t.Helper()
	var qs []*Question
	parseObject(t, &qs, jsonFmt, jsonArgs...)
	return qs
}

func ParseUserAnswer(t *testing.T, jsonFmt string, jsonArgs ...interface{}) *UserAnswer {
	t.Helper()
	var u UserAnswer
	parseObject(t, &u, jsonFmt, jsonArgs...)
	return &u
}

func ParseLeaderboard(t *testing.T, jsonFmt string, jsonArgs ...interface{}) *Leaderboard {
	t.Helper()
	var lb Leaderboard
	parseObject(t, &lb, jsonFmt, jsonArgs...)
	return &lb
}
