package model

import (
	"database/sql"
	"time"
)

type User struct {
	ID       int    `db:"id"         json:"id"`
	OpaqueID string `db:"opaque_id"  json:"opaque_id"`
	TwitchID string `db:"twitch_id"  json:"-"`
	Username string `db:"username"   json:"username"`

	db *DB
}

type UserAnswer struct {
	UserID     int       `db:"user_id"      json:"user_id"`
	User       *User     `db:"-"            json:"user"`
	QuestionID int       `db:"question_id"  json:"question_id"`
	Question   *Question `db:"-"            json:"-"`
	AnswerID   *int      `db:"answer_id"    json:"answer_id"`
	Answer     *Answer   `db:"-"            json:"-"`
	Message    *string   `db:"message"      json:"message,omitempty"`

	db *DB
}

func (db *DB) FindUser(id int) (*User, Error) {
	var user User
	if err := db.Get(&user, `SELECT * FROM "users" WHERE "id" = $1 LIMIT 1`, id); err == sql.ErrNoRows {
		return nil, nil
	} else if err != nil {
		return nil, DBError(err)
	} else {
		user.db = db
		return &user, nil
	}
}

func (db *DB) FindOrCreateUser(opaqueID string) (*User, Error) {
	var user User
	err := db.Get(&user,
		`INSERT INTO "users" ("opaque_id", "twitch_id", "username") VALUES ($1, $2, $3) ON CONFLICT DO NOTHING RETURNING *`,
		opaqueID, "twitch:"+opaqueID, "user:"+opaqueID,
	)
	if err == sql.ErrNoRows {
		// When upsert finds existing row, "DO NOTHING" makes it return 0 rows despite "RETURNING *".
		err = db.Get(&user, `SELECT * FROM "users" WHERE "opaque_id" = $1 LIMIT 1`, opaqueID)
	}
	if err != nil {
		return nil, DBError(err)
	}
	user.db = db
	return &user, nil
}

func (question *Question) AddUserAnswer(opaqueID string, answerID int, skipTimeCheck bool) (*UserAnswer, Error) {
	db := question.db
	user, dbErr := db.FindOrCreateUser(opaqueID)
	if dbErr != nil {
		return nil, dbErr
	}

	answer := question.FindAnswer(answerID)
	if answer == nil {
		return nil, UserErrorf("answer %d is not an answer to question %d", answerID, question.ID)
	}

	if !skipTimeCheck {
		if question.ActiveFrom == nil || question.ActiveUntil == nil {
			return nil, UserErrorf("question %d has never been active", question.ID)
		}
		now := time.Now().UTC()
		if question.ActiveFrom.After(now) {
			return nil, UserErrorf("question %d is not active yet", question.ID)
		} else if question.ActiveUntil.Before(now) {
			return nil, UserErrorf("question %d is no longer active", question.ID)
		}
	}

	// Full transaction:
	// 1. Upsert answer. If user already answered, ignore replays of same answer or complain if different.
	// 2. Increment response counts for answer and question.
	// 3. Update user's record in the channel's leaderboard.

	tx, err := db.Beginx()
	if err != nil {
		return nil, DBErrorf("error initializing transaction: %s", err)
	}
	defer tx.Rollback()

	// 1. Upsert answer.
	userAnswer := UserAnswer{
		UserID:     user.ID,
		User:       user,
		QuestionID: question.ID,
		Question:   question,
		AnswerID:   &answerID,
		Answer:     answer,
		Message:    nil,
	}
	err = tx.Get(&userAnswer,
		`INSERT INTO "user_answers" ("question_id", "user_id", "answer_id") VALUES ($1, $2, $3) ON CONFLICT DO NOTHING RETURNING *`,
		question.ID, user.ID, answerID,
	)
	if err == sql.ErrNoRows {
		// UPSERT returns no rows on success. x_x
		err = tx.Get(&userAnswer,
			`SELECT * FROM "user_answers" WHERE "question_id" = $1 AND "user_id" = $2 LIMIT 1`,
			question.ID, user.ID)
	}
	if err != nil {
		return nil, DBErrorf("error adding user answer: %s", err)
	}

	// If the record already existed, the DB fetch updates the `AnswerID` field, but not `Answer`.
	if *userAnswer.AnswerID != answerID {
		if userAnswer.Answer = question.FindAnswer(*userAnswer.AnswerID); userAnswer.Answer == nil {
			return nil, DBErrorf("user's existing answer was not an answer to this question: %d", *userAnswer.AnswerID)
		}
		return &userAnswer, UserErrorf("user has already answered this question")
	}

	// 2. Increment response counts.
	if _, err = tx.Exec(`UPDATE "questions" SET "responses" = "responses" + 1 WHERE "id" = $1`, question.ID); err != nil {
		return nil, DBErrorf("error updating response count for question %d: %s", question.ID, err)
	} else if _, err = tx.Exec(`UPDATE "answers" SET "responses" = "responses" + 1 WHERE "id" = $1`, answerID); err != nil {
		return nil, DBErrorf("error updating response count for answer %d: %s", answerID, err)
	}

	// 3. Update user's record in the channel's leaderboard.
	if question.HasCorrect() {
		correct := 0
		if answer.IsCorrect {
			correct = 1
		}
		if _, err = tx.Exec(
			`INSERT INTO "leaderboard" ("channel_id", "user_id", "correct", "answered", "percent") VALUES ($1, $2, $3, 1, $4) `+
				`ON CONFLICT ("channel_id", "user_id") DO UPDATE SET ("correct", "answered", "percent") = `+
				`("leaderboard"."correct"+$3, "leaderboard"."answered"+1, ("leaderboard"."correct" + $3)::REAL / ("leaderboard"."answered" + 1)::REAL)`,
			question.ChannelID, user.ID, correct, float32(correct),
		); err != nil {
			return nil, DBErrorf("error updating leaderboard: %s", err)
		}
	}

	if err = tx.Commit(); err != nil {
		return nil, DBErrorf("error committing transaction: %s", err)
	}

	return &userAnswer, nil
}

func (q *Question) ForUser(opaqueID string) Error {
	user, qErr := q.db.FindOrCreateUser(opaqueID)
	if qErr != nil {
		return qErr
	}

	ua := UserAnswer{
		UserID:     user.ID,
		User:       user,
		QuestionID: q.ID,
		Question:   q,
		AnswerID:   nil,
		Answer:     nil,
		Message:    nil,
	}

	var message sql.NullString
	var answerID int
	answered := false
	if err := q.db.QueryRowx(
		`SELECT "answer_id", "message" FROM "user_answers" WHERE "question_id" = $1 AND "user_id" = $2 LIMIT 1`,
		q.ID, user.ID,
	).Scan(&answerID, &message); err == sql.ErrNoRows {
		q.Answered = &answered
		q.UserAnswer = &ua
		return nil
	} else if err != nil {
		return DBError(err)
	} else if message.Valid {
		ua.Message = &message.String
	}

	answered = true
	ua.AnswerID = &answerID
	if ua.Answer = q.FindAnswer(*ua.AnswerID); ua.Answer == nil {
		return DBErrorf("user %d's existing answer (%d) is not an answer to question %d", user.ID, ua.AnswerID, q.ID)
	}

	q.Answered = &answered
	q.UserAnswer = &ua
	return nil
}
