// +build test

package model

import (
	"fmt"
	"testing"
	"time"

	uuid "github.com/satori/go.uuid"
)

func TestUser(t *testing.T) {
	db := InitTestDB(t)
	defer db.Close()

	t.Run("FindUser", findUserTests(db))
	t.Run("FindOrCreateUser", findOrCreateUserTests(db))
	t.Run("AddUserAnswer", addUserAnswerTests(db))
	t.Run("QuestionForUser", questionForUserTests(db))
}

func findUserTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		t.Run("DoesNotExist", func(t *testing.T) {
			if user, err := db.FindUser(-1); err != nil {
				t.Fatalf("unexpected error: %s", err)
			} else if user != nil {
				t.Fatalf("unexpected success: ID %d, opaque %q", user.ID, user.OpaqueID)
			}
		})

		t.Run("Exists", func(t *testing.T) {
			user := db.NewTestUser(t, nil)
			if u, err := db.FindUser(user.ID); err != nil {
				t.Fatalf("unexpected error: %s", err)
			} else if u == nil {
				t.Fatalf("unexpected nil user; expected user with ID %d / opaque ID %q", user.ID, user.OpaqueID)
			} else {
				u.AssertEqual(t, user)
			}
		})
	}
}

func findOrCreateUserTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		t.Run("DoesNotExist", func(t *testing.T) {
			opaqueID := uuid.NewV4().String()
			if user, err := db.FindOrCreateUser(opaqueID); err != nil {
				t.Fatalf("unexpected error creating user %q: %s", opaqueID, err)
			} else if user == nil {
				t.Fatalf("unexpected nil user; expected user with opaque ID %q", opaqueID)
			} else if user.OpaqueID != opaqueID {
				t.Fatalf("wrong user: opaque ID %q; expected %q", opaqueID)
			}
		})

		t.Run("Exists", func(t *testing.T) {
			user := db.NewTestUser(t, nil)
			if u, err := db.FindOrCreateUser(user.OpaqueID); err != nil {
				t.Fatalf("unexpected error: %s", err)
			} else if u == nil {
				t.Fatalf("unexpected nil user; expected user with ID %d / opaque ID %q", user.ID, user.OpaqueID)
			} else {
				u.AssertEqual(t, user)
			}
		})
	}
}

func addUserAnswerTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		question := db.NewTestQuestion(t, nil)
		answer := question.Answers[0]
		user := db.NewTestUser(t, nil)

		t.Run("BadAnswer", func(t *testing.T) {
			if ua, err := question.AddUserAnswer(user.OpaqueID, -1, false); err == nil || ua != nil {
				t.Fatalf("unexpected success / non-nil reply: %#v", ua)
			} else if err.Error() != fmt.Sprintf("answer -1 is not an answer to question %d", question.ID) {
				t.Fatalf("wrong error message: %q", err.Error())
			}
		})

		t.Run("InactiveAnswer", func(t *testing.T) {
			if ua, err := question.AddUserAnswer(user.OpaqueID, answer.ID, false); err == nil || ua != nil {
				t.Fatalf("unexpected success / non-nil reply: %#v", ua)
			} else if err.Error() != fmt.Sprintf("question %d has never been active", question.ID) {
				t.Fatalf("wrong error message: %q", err.Error())
			}
		})

		now := time.Now().UTC()
		yesterday := now.Add(-24 * time.Hour)
		tomorrow := now.Add(+24 * time.Hour)

		t.Run("LateAnswer", func(t *testing.T) {
			q := db.NewTestQuestion(t, Attr{"ActiveFrom": &yesterday, "ActiveUntil": &yesterday})
			aID := q.AnswerIDs[0]
			if ua, err := q.AddUserAnswer(user.OpaqueID, aID, false); err == nil || ua != nil {
				t.Fatalf("unexpected success / non-nil reply: %#v", ua)
			} else if err.Error() != fmt.Sprintf("question %d is no longer active", q.ID) {
				t.Fatalf("wrong error message: %q", err.Error())
			}
		})

		t.Run("EarlyAnswer", func(t *testing.T) {
			q := db.NewTestQuestion(t, Attr{"ActiveFrom": &tomorrow, "ActiveUntil": &tomorrow})
			aID := q.AnswerIDs[0]
			if ua, err := q.AddUserAnswer(user.OpaqueID, aID, false); err == nil || ua != nil {
				t.Fatalf("unexpected success / non-nil reply: %#v", ua)
			} else if err.Error() != fmt.Sprintf("question %d is not active yet", q.ID) {
				t.Fatalf("wrong error message: %q", err.Error())
			}
		})

		t.Run("OK", func(t *testing.T) {
			q := db.NewTestQuestion(t, Attr{"ActiveFrom": &yesterday, "ActiveUntil": &tomorrow})
			aID := q.AnswerIDs[0]
			if ua, err := q.AddUserAnswer(user.OpaqueID, aID, false); err != nil {
				t.Fatalf("unexpected error adding user %q answer %d to question %d: %s", user.OpaqueID, aID, q.ID, err)
			} else if ua.UserID != user.ID {
				t.Fatalf("wrong user ID in UserAnswer: %d; expected %d", ua.UserID, user.ID)
			} else if ua.QuestionID != q.ID {
				t.Fatalf("wrong question ID in UserAnswer: %d; expected %d", ua.QuestionID, q.ID)
			} else if *ua.AnswerID != aID {
				t.Fatalf("wrong answer ID in UserAnswer: %d; expected %d", *ua.AnswerID, aID)
			}
		})

		t.Run("Duplicate", func(t *testing.T) {
			q := db.NewTestQuestion(t, Attr{"Active": true})
			oID := user.OpaqueID
			aID0, aID1 := q.AnswerIDs[0], q.AnswerIDs[1]
			if _, err := q.AddUserAnswer(oID, aID0, false); err != nil {
				t.Fatalf("unexpected error adding user %q answer %d to question %d: %s", oID, aID0, q.ID, err)
			} else if ua, err := q.AddUserAnswer(oID, aID1, false); err == nil {
				t.Fatalf("unexpected success: %#v", ua)
			} else if err.Error() != "user has already answered this question" {
				t.Fatalf("wrong error message: %q", err.Error())
			} else if *ua.AnswerID != aID0 {
				t.Fatalf("wrong existing answer ID: %d; expected %d", *ua.AnswerID, aID0)
			}
		})
	}
}

func questionForUserTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		user := db.NewTestUser(t, nil)

		t.Run("NoAnswer", func(t *testing.T) {
			q := db.NewTestQuestion(t, nil)
			uaExp := fmtPtr(&UserAnswer{
				UserID:     user.ID,
				QuestionID: q.ID,
				AnswerID:   nil,
			})
			if err := q.ForUser(user.OpaqueID); err != nil {
				t.Fatalf("unexpected error: %s", err)
			} else if q.UserAnswer == nil {
				t.Fatal("expected Question.UserAnswer to be non-nil")
			} else if uaAct := fmtPtr(q.UserAnswer); uaAct != uaExp {
				t.Fatalf("wrong UserAnswer: %s; expected %s", uaAct, uaExp)
			}
		})

		t.Run("WithAnswer", func(t *testing.T) {
			q := db.NewTestQuestion(t, nil)
			answerID := q.AnswerIDs[0]
			uaExpStr := fmtPtr(&UserAnswer{
				UserID:     user.ID,
				QuestionID: q.ID,
				AnswerID:   &answerID,
			})
			if uaAct, err := q.AddUserAnswer(user.OpaqueID, answerID, true); err != nil {
				t.Fatalf("unexpected error adding answer: %s", err.Error())
			} else if uaActStr := fmtPtr(uaAct); uaActStr != uaExpStr {
				t.Fatalf("wrong UserAnswer reply from user.Answer: %s\nexpected: %s", uaActStr, uaExpStr)
			} else if q, err := db.FindQuestion(q.ID); err != nil {
				t.Fatalf("unexpected error when re-fetching question: %s", err)
			} else if err = q.ForUser(user.OpaqueID); err != nil {
				t.Fatalf("unexpected error getting question for user: %s", err)
			} else if uaActStr = fmtPtr(q.UserAnswer); uaActStr != uaExpStr {
				t.Fatalf("wrong UserAnswer: %s; expected %s", uaActStr, uaExpStr)
			}
		})
	}
}
