package main

import (
	"flag"
	"fmt"
	"os"

	"ting/handlers"
	"ting/logging"
	"ting/model"

	"github.com/gin-gonic/gin"
	"github.com/sirupsen/logrus"
)

func startupMessage(cfg Config) {
	if cfg.Logging.Format == "json" {
		logrus.WithFields(map[string]interface{}{
			"version": VERSION,
			"config":  cfg,
		}).Info("starting")
	} else {
		logrus.Info("ting " + VERSION)
		logrus.Info("configuration:")
		logrus.Infof("    API port: %d", cfg.Port)
		logrus.Infof("    Database URL: %q", cfg.DbUrl)

		switch {
		case cfg.JWT.Disabled:
			logrus.Info("    JWT: disabled")
		case cfg.JWT.ExtensionID != "":
			logrus.Infof("    JWT: fetching by ExtensionID = %q", cfg.JWT.ExtensionID)
		case cfg.JWT.Secret != "":
			logrus.Info("    JWT: provided as CLI arg")
		case cfg.JWT.SecretFile != "":
			logrus.Infof("    JWT: reading from %q", cfg.JWT.SecretFile)
		}

		if cfg.TLS.Enabled() {
			logrus.Info("    TLS:")
			logrus.Infof("        CertFile: %q", cfg.TLS.CertFile)
			logrus.Infof("        KeyFile:  %q", cfg.TLS.KeyFile)
		} else {
			logrus.Info("    TLS: disabled")
		}

		logrus.Info("logging:")
		logrus.Infof("    format: %q", cfg.Logging.Format)
		logrus.Infof("    level: %q", cfg.Logging.Level)
	}
}

func main() {
	cfg, err := ParseArgv(os.Args[1:])
	if err == flag.ErrHelp {
		os.Exit(0)
	} else if err != nil {
		fmt.Fprintln(os.Stderr, err.Error())
		os.Exit(1)
	}

	if err = logging.Init(cfg.Logging.Format, cfg.Logging.Level); err != nil {
		fmt.Fprintf(os.Stderr, "error initializing logging: %s\n", err)
		os.Exit(1)
	}

	startupMessage(cfg)

	var jwtMW gin.HandlerFunc
	if !cfg.JWT.Disabled {
		if secret, err := cfg.JWT.GetSecret(); err != nil {
			logrus.Fatalf("error obtaining JWT shared secret: %s", err)
		} else if jwtMW, err = handlers.JWTMiddleware(secret); err != nil {
			logrus.Fatal(err.Error())
		}
	}

	db, err := model.InitDB(cfg.DbUrl)
	if err != nil {
		logrus.Fatalf("error initializing db client: %s\n", err)
	}

	gin.SetMode("release") // only way to stop gin from pooping all over stdout
	engine := gin.New()
	engine.Use(LogRequestMW, RecoverMW)

	// Add janky CORS middleware.
	// TODO: https://github.com/gin-contrib/cors
	engine.Use(func(ctx *gin.Context) {
		ctx.Header("Access-Control-Allow-Origin", "*")
		ctx.Header("Access-Control-Allow-Methods", ctx.Request.Method)
		ctx.Next()
	})

	// Register non-JWT endpoints.
	engine.GET("/health", func(ctx *gin.Context) { ctx.Status(200) })
	engine.GET("/version", func(ctx *gin.Context) { ctx.String(200, VERSION) })
	handlers.Register(engine, db)

	// Register JWT endpoints.
	if jwtMW != nil {
		jwtGroup := engine.Group("/jwt", jwtMW)
		handlers.Register(jwtGroup, db)
	}

	addr := fmt.Sprintf("0.0.0.0:%d", cfg.Port)
	if cfg.TLS.Enabled() {
		err = engine.RunTLS(addr, cfg.TLS.CertFile, cfg.TLS.KeyFile)
	} else {
		err = engine.Run(addr)
	}
	logrus.Fatalf("error starting HTTP server: %s", err)
}
