// +build test

package util

import (
	"bytes"
	"encoding/json"
	"fmt"
	"reflect"
	"sort"
	"strconv"
	"testing"
	"time"
)

func AtoiT(t *testing.T, s string) (x int) {
	t.Helper()
	var err error
	if x, err = strconv.Atoi(s); err != nil {
		t.Fatalf("failed to parse string as integer: %q", s)
	}
	return x
}

func AtofT(t *testing.T, s string) (x float64) {
	t.Helper()
	var err error
	if x, err = strconv.ParseFloat(s, 64); err != nil {
		t.Fatalf("failed to parse string as float: %q", s)
	}
	return x
}

func deref(x interface{}) (reflect.Value, reflect.Type) {
	v := reflect.ValueOf(x)
	if t := reflect.TypeOf(x); t.Kind() != reflect.Ptr || v.IsNil() {
		return v, t
	} else {
		return reflect.Indirect(v), t.Elem()
	}
}

func AssertEqual(t *testing.T, v1 interface{}, v2 interface{}) {
	t.Helper()
	assertEqual(t, v1, v2, "")
}

func assertEqual(t *testing.T, v1I interface{}, v2I interface{}, errPrefix string) {
	t.Helper()
	if v1I == nil && v2I == nil {
		return
	} else if XorNil(v1I, v2I) {
		t.Fatalf(errPrefix+"mismatched nils: %s vs %s", NilStr(v1I), NilStr(v2I))
	}

	// Do one level of pointer dereferencing to compare values,
	// and to allow convenient check of a pointer's value, e.g. `AssertEqual(t, strPtr, "foo")`.
	v1, t1 := deref(v1I)
	v2, t2 := deref(v2I)
	if t1 != t2 {
		t.Fatalf(errPrefix+"mismatched types: %s vs %s", t1, t2)
	} else if t1.Kind() == reflect.Slice {
		if v1.Len() != v2.Len() {
			t.Fatalf(errPrefix+"mismatched lengths: %d vs %d", v1.Len(), v2.Len())
		}
		for i := 0; i < v1.Len(); i++ {
			assertEqual(t, v1.Index(i).Interface(), v2.Index(i).Interface(), fmt.Sprintf("[%d]: ", i))
		}
		return
	} else if t1.Kind() == reflect.Map {
		if v1.Len() != v2.Len() {
			t.Fatalf(errPrefix+"mismatched lengths: %d vs %d", v1.Len(), v2.Len())
		}
		for _, kV := range v1.MapKeys() {
			vv2 := v2.MapIndex(kV)
			if !vv2.IsValid() {
				t.Fatalf(errPrefix+"[%#v]: not found in right map", kV.Interface())
			}
			assertEqual(t, v1.MapIndex(kV).Interface(), vv2.Interface(), fmt.Sprintf("[%#v]: ", kV.Interface()))
		}
		return
	}

	if !t1.Comparable() {
		t.Fatalf(errPrefix+"type %s is not comparable", t1)
	} else if v1.Interface() != v2.Interface() {
		t.Fatalf(errPrefix+"values don't match: (%s) %#v vs (%s) %#v", t1.String(), v1.Interface(), t2.String(), v2.Interface())
	}
}

// Returned by `StructDiff`.
type DiffField struct {
	Name string
	A, B interface{}
}

// Like `reflect.DeepEqual`, but accepts only structs and describes differences.
func StructDiff(aArg, bArg interface{}) (diffs []DiffField) {
	a := reflect.ValueOf(aArg)
	b := reflect.ValueOf(bArg)
	if a.Type() != b.Type() {
		panic(fmt.Sprintf("arguments are not the same type: %T vs %T", aArg, bArg))
	} else if a.Kind() != reflect.Struct {
		panic(fmt.Sprintf("arguments are not structs: %T", aArg))
	}

	t := a.Type()
	for i := 0; i < t.NumField(); i++ {
		field := t.Field(i)
		// `reflect` won't let us get values of unexported fields. :(
		if c := field.Name[0]; c >= 'a' && c <= 'z' {
			continue
		}
		aValue := a.Field(i).Interface()
		bValue := b.Field(i).Interface()
		if !reflect.DeepEqual(aValue, bValue) {
			diffs = append(diffs, DiffField{field.Name, aValue, bValue})
		}
	}
	return diffs
}

// Convenience method that formats a test failure message based on `StructDiff`'s output.
func DiffErrorMessage(diffs []DiffField, indent int) string {
	indentStr := ""
	for i := 0; i < indent; i++ {
		indentStr += "    "
	}
	buf := bytes.NewBufferString(fmt.Sprintf("%d fields differ:\n", len(diffs)))
	for _, diff := range diffs {
		buf.WriteString(fmt.Sprintf("%s%q (%T): %#v vs %#v\n", indentStr, diff.Name, diff.A, diff.A, diff.B))
	}
	return buf.String()
}

func sortIntsCopy(xs []int) []int {
	if xs == nil {
		return nil
	}
	ys := make([]int, len(xs))
	for i, x := range xs {
		ys[i] = x
	}
	sort.Ints(ys)
	return ys
}

func IntsSortEqual(xs, ys []int) bool {
	if (xs == nil && ys != nil) || (xs != nil && ys == nil) {
		return false
	} else if xs == nil && ys == nil {
		return true
	} else if len(xs) != len(ys) {
		return false
	}
	xsSorted := sortIntsCopy(xs)
	ysSorted := sortIntsCopy(ys)
	for i := 0; i < len(xsSorted); i++ {
		if xsSorted[i] != ysSorted[i] {
			return false
		}
	}
	return true
}

func ToJSON(t *testing.T, v interface{}) (s string) {
	t.Helper()
	if buf, err := json.Marshal(v); err != nil {
		t.Fatalf("error serializing %T to JSON: %s\nfull value = %#v", v, err, v)
	} else {
		s = string(buf)
	}
	return s
}

func ToJSONPP(t *testing.T, v interface{}) (s string) {
	t.Helper()
	if buf, err := json.MarshalIndent(v, "", "    "); err != nil {
		t.Fatalf("error serializing %T to JSON: %s\nfull value = %#v", v, err, v)
	} else {
		s = string(buf)
	}
	return s
}

func FromJSON(t *testing.T, s string, obj interface{}) {
	t.Helper()
	if err := json.Unmarshal([]byte(s), &obj); err != nil {
		t.Fatalf("error deserializing JSON: %s\ninput: %s", err, s)
	}
}

func MapFromJSON(t *testing.T, s string) (obj map[string]interface{}) {
	t.Helper()
	FromJSON(t, s, &obj)
	return obj
}

// Mapf<T,U>(f: func(T)U, xs: []T) -> []U
// Result is obviously boxed in empty interface; just cast to `[]U` after call.
func Mapf(f, xs interface{}) interface{} {
	xsV := reflect.ValueOf(xs)
	fV := reflect.ValueOf(f)
	yT := fV.Type().Out(0)
	ysV := reflect.MakeSlice(reflect.SliceOf(yT), xsV.Len(), xsV.Len())
	for i := 0; i < xsV.Len(); i++ {
		ysV.Index(i).Set(fV.Call([]reflect.Value{xsV.Index(i)})[0])
	}
	return ysV.Interface()
}

func Timestamp(hours int) string {
	return time.Now().Add(time.Duration(hours) * time.Hour).UTC().Format(time.RFC3339)
}
