package util

import (
	"bytes"
	"fmt"
	"reflect"
	"time"

	"ting/util/set"
)

var nilableTypes set.StringSet

func init() {
	// `reflect.IsNil()`: "argument must be a chan, func, interface, map, pointer, or slice value"
	nilableTypes = set.NewStringSet("chan", "func", "interface", "map", "ptr", "slice")
}

// Safe wrapper around `reflect.IsNil()` that just returns `false` for types that can't be `nil` instead of panicking.
func IsNil(v interface{}) bool {
	if v == nil {
		return true
	} else if kind := reflect.TypeOf(v).Kind(); nilableTypes.Has(kind.String()) {
		return reflect.ValueOf(v).IsNil()
	}
	return false
}

func NilStr(v interface{}) string {
	if IsNil(v) {
		return "nil"
	}
	return "non-nil"
}

func Xor(x, y bool) bool {
	return (x && !y) || (!x && y)
}

func XorNil(a, b interface{}) bool {
	aNil, bNil := IsNil(a), IsNil(b)
	return Xor(aNil, bNil)
}

func ParseTimep(x interface{}) (*time.Time, error) {
	if x == nil {
		return nil, nil
	} else if ts, ok := x.(string); !ok {
		return nil, fmt.Errorf("expected string; found %T", x)
	} else if ts == "" {
		return nil, nil
	} else if t, err := time.Parse(time.RFC3339, ts); err != nil {
		return nil, err
	} else {
		utc := t.UTC()
		return &utc, nil
	}
}

func FmtStrings(strs []string) string {
	if len(strs) == 0 {
		return ""
	}
	buf := new(bytes.Buffer)
	buf.WriteString(fmt.Sprintf("%q", strs[0]))
	for i := 1; i < len(strs); i++ {
		buf.WriteString(fmt.Sprintf(", %q", strs[i]))
	}
	return buf.String()
}
