variable "service_name" { default = "ting-backend" }
variable "service_env" { default = "staging" }

provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-dev/"
}

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "ting-backend" {
  name = "ting-backend"
}

data "template_file" "ting_backend_task_definition" {
  template = "${file("task-definitions/ting-backend.json")}"
  vars {
    name = "ting-backend"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.ting-backend.repository_url, "https://", "")}:latest"
    database_url = "postgres://${aws_db_instance.default.username}:${aws_db_instance.default.password}@${aws_db_instance.default.endpoint}/${aws_db_instance.default.name}"
    container_port = 3000
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "ting_backend_task" {
  family = "ting-backend"
  container_definitions = "${data.template_file.ting_backend_task_definition.rendered}"
}

module "service" {
  # source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-09-25"
  source = "/home/mr/dev/esports/terraform/modules/service"
  name = "ting-backend"
  prefix = "ting_backend"
  container_port = "3000"
  alb = true
  alb_health_uri = "/health"
  cluster = "programs"
  env = "staging"
  desired_count = 1
  # staging.rlcs.gg us-west-2
  # certificate_arn = "arn:aws:acm:us-west-2:157383839067:certificate/1dc70f4b-e723-49fd-bb88-c5f5faba9a36"
  task_definition = "${aws_ecs_task_definition.ting_backend_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.dev.alarm_sns_topic_arn}"
}

output "ecr_url" {
  value = "${aws_ecr_repository.ting-backend.repository_url}"
}
