import { config } from 'config';
import {
  Editor,
  EditorState,
  Entity,
  RichUtils,
  ContentState,
  CompositeDecorator,
  AtomicBlockUtils,
  convertFromRaw,
} from 'draft-js';
import * as React from 'react';
import { connect } from 'react-redux';
import { uploadImage } from 'tourney-sdk-react';

import { ImageComponent } from './image-component';
import { InlineToolbar } from './inline-toolbar';
import { SideToolbar } from './side-toolbar';

require('./styles.scss');

export const getSelectionRange = () => {
  const selection = window.getSelection();
  if (selection.rangeCount === 0){
    return null;
  }
  return selection.getRangeAt(0);
};

export const getSelectedBlockElement = (range) => {
  let node = range.startContainer;
  do {
    const nodeIsDataBlock = node.getAttribute
                            ? node.getAttribute('data-block')
                            : null;
    if (nodeIsDataBlock) {
      return node;
    }
    node = node.parentNode;
  } while (node !== null);
  return null;
};

export const getSelectionCoords = (selectionRange) => {
  const editorBounds = document.getElementById('richEditor').getBoundingClientRect();
  const rangeBounds = selectionRange.getBoundingClientRect();
  const rangeWidth = rangeBounds.right - rangeBounds.left;
  const rangeHeight = rangeBounds.bottom - rangeBounds.top;
  const offsetLeft = (rangeBounds.left - editorBounds.left)
            + (rangeWidth / 2)
            /* 72px is width of inline toolbar */
            - (72 / 2);
  // 42px is height of inline toolbar (35px) + 5px center triangle and 2px for spacing
  const offsetTop = rangeBounds.top - editorBounds.top - 42;
  return { offsetLeft, offsetTop };
};

interface RichEditorProps {
  onUpdate: (EditorState) => {};
  intialState: EditorState;
}

export class RichEditorComponent extends React.Component<any, any> {
  public constructor(props) {
    super(props);

    this.state = {
      editorState: EditorState.createWithContent(convertFromRaw(props.initialState)),
      inlineToolbar: { show: false }
    };

    this.onChange = (editorState) => {
      if (!editorState.getSelection().isCollapsed()) {
        const selectionRange = getSelectionRange();
        const selectionCoords = getSelectionCoords(selectionRange);
        this.setState({
          inlineToolbar: {
            show: true,
            position: {
              top: selectionCoords.offsetTop,
              left: selectionCoords.offsetLeft
            }
          }
        });
      } else {
        this.setState({ inlineToolbar: { show: false } });
      }

      this.setState({ editorState });
      this.props.onUpdate(editorState);
      setTimeout(this.updateSelection, 0);
    }

    this.focus = () => this.refs.editor.focus();
    this.updateSelection = () => this._updateSelection();
    this.handleKeyCommand = (command) => this._handleKeyCommand(command);
    this.handleFileInput = (e) => this._handleFileInput(e);
    this.handleUploadImage = () => this._handleUploadImage();
    this.toggleBlockType = (type) => this._toggleBlockType(type);
    this.toggleInlineStyle = (style) => this._toggleInlineStyle(style);
    this.insertImage = (file) => this._insertImage(file);
    this.blockRenderer = (block) => {
      if (block.getType() === 'atomic') {
        return {
          component: ImageComponent
        };
      }
      return null;
    }
    this.blockStyler = (block) => {
      if (block.getType() === 'unstyled') {
        return 'paragraph';
      }
      return null;
    }
  }

  _updateSelection() {
    const selectionRange = getSelectionRange();
    let selectedBlock;
    if (selectionRange) {
      selectedBlock = getSelectedBlockElement(selectionRange);
    }
    this.setState({
      selectedBlock,
      selectionRange
    });
  }

  _handleKeyCommand(command) {
    const { editorState } = this.state;
    const newState = RichUtils.handleKeyCommand(editorState, command);
    if (newState) {
      this.onChange(newState);
      return true;
    }
    return false;
  }

  _toggleBlockType(blockType) {
    this.onChange(
      RichUtils.toggleBlockType(
        this.state.editorState,
        blockType
      )
    );
  }

  _toggleInlineStyle(inlineStyle) {
    this.onChange(
      RichUtils.toggleInlineStyle(
        this.state.editorState,
        inlineStyle
      )
    );
  }

  _insertImage(url) {
    const entityKey = Entity.create('atomic', 'IMMUTABLE', { src: url });
		this.onChange(AtomicBlockUtils.insertAtomicBlock(
      this.state.editorState,
      entityKey,
      ' '
    ));
  }

  _handleFileInput(e) {
    const fileList = e.target.files;
    const file = fileList[0];
    this.props.dispatch(uploadImage(file))
      .then((result: any) => {
        // result
        this.insertImage(`${config.apiUrl}/v1/images/${result.id}`);
      }
  }

  _handleUploadImage() {
    this.refs.fileInput.click();
  }

  render() {
    const { editorState, selectedBlock, selectionRange } = this.state;
    let sideToolbarOffsetTop = 0;

    let selectedBlockElement: JSX.Element;
    let inlineToolbar: JSX.Element;

    if (selectedBlock) {
      const editor = document.getElementById('richEditor');
      const editorBounds = editor.getBoundingClientRect();
      const blockBounds = selectedBlock.getBoundingClientRect();

      sideToolbarOffsetTop = (blockBounds.bottom - editorBounds.top)
                           - selectedBlock.clientHeight; // height of side toolbar
      selectedBlockElement = <SideToolbar
        editorState={editorState}
        style={{ top: sideToolbarOffsetTop }}
        onToggle={this.toggleBlockType}
        onUploadImage={this.handleUploadImage}
      />;
    }

    return (
      <div className="editor" id="richEditor" onClick={this.focus}>
        {selectedBlockElement}
        {this.state.inlineToolbar.show
          ? <InlineToolbar
              editorState={editorState}
              onToggle={this.toggleInlineStyle}
              position={this.state.inlineToolbar.position}
            />
          : null
        }
        <Editor
          blockRendererFn={this.blockRenderer}
          blockStyleFn={this.blockStyler}
          editorState={editorState}
          handleKeyCommand={this.handleKeyCommand}
          onChange={this.onChange}
          placeholder="Write something..."
          spellCheck={true}
          readOnly={this.state.editingImage}
          ref="editor"
        />
        <input type="file" ref="fileInput" style={{display: 'none'}}
          onChange={this.handleFileInput} />
      </div>
    );
  }
}

export const RichEditor = connect()(RichEditorComponent);