import * as _ from 'lodash';
import * as React from 'react';
import {
  Form,
  Label,
  Input,
  Dropdown,
  Checkbox
} from 'semantic-ui-react';
import * as Datetime from 'react-datetime';
import * as moment from 'moment';
import { BlockPicker } from 'react-color';

export interface IFormFieldProps extends React.Props<FormField> {
  type: string;
  value: any;
  name: string;
  options?: Array<Object>;
  placeholder?: string;
  errors?: boolean | JSX.Element;
  onChange?: any;
}

export class FormField extends React.Component<IFormFieldProps, any> {
  public render() {
    let input;
    let { name, type, value } = this.props;
    switch (type) {
      case 'color':
        input = <BlockPicker
          color={value}
          onChange={(value: any, event: any) =>
            this.props.onChange(event, { ...this.props,
                                value: `rgba(${Object.values(value.rgb).join(',')})` })}
        />
        break;
      case 'checkbox':
        input = <Checkbox
          toggle
          name={this.props.name}
          checked={this.props.value ? true : false}
          value='1'
          onChange={this.props.onChange}
        />;
        break;
      case 'select-multiple':
      case 'select':
        input = <Dropdown
          multiple={type === 'select-multiple' ? true : false}
          fluid
          selection
          search
          name={this.props.name}
          value={this.props.value}
          placeholder={this.props.placeholder}
          options={this.props.options}
          onChange={this.props.onChange}
        />;
        break;
      case 'image-upload':
        const preview = this.props.value && this.props.value.original !== ''
          ? <img className="image-upload__image" src={this.props.value.preview} />
          : <div className="image-upload__empty">No Image Uploaded</div>;
        input = (
          <div className="image-upload__container">
            {preview}
            <Input
              type='file'
              name={this.props.name}
              onChange={this.props.onChange}
            />
        </div>);
        break;
      case 'file':
        input = <Input
          type='file'
          name={this.props.name}
          onChange={this.props.onChange}
        />;
        break;
      // WHATWG and W3C can suck it :(
      case 'datetime-local':
        const date = new Date(value);
        input = <Datetime
          value={date}
          defaultValue={date}
          dateFormat="ddd, MM/DD/YY - "
          timeFormat="h:mmA"
          utc={false}
          input={true}
          onChange={(value: any) =>
            this.props.onChange({
              target: {
                type: 'datetime-local'
              }
            }, { ...this.props, value: value })
          }
        />;
        break;
      default:
        input = <Input
          type={this.props.type || 'text'}
          name={this.props.name}
          placeholder={this.props.placeholder}
          value={value}
          onChange={this.props.onChange}
        />;
    }

    return <Form.Field error={this.props.errors ? true : false}>
      <label>{this.props.name}</label>
      {input}
      {this.props.errors}
    </Form.Field>;
  }
}
