import { EditorState, convertToRaw } from 'draft-js';
import * as _ from 'lodash';
import * as moment from 'moment';
import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { connect } from 'react-redux';
import { singularize } from 'inflection';

import * as SUI from 'semantic-ui-react';
import JSONTree from 'react-json-tree';
import {
  Editor,
  createEditorState,
} from 'medium-draft';

import { FormField } from './field';
import { MediumEditor } from '../medium-editor';
import { denormalize } from 'tourney-sdk-react';

import 'medium-draft/lib/index.css';

const style = require('./styles.scss');

let mapDispatchToProps = (dispatch: any) => {
  return { dispatch };
};

let mapStateToProps = (state: any, ownProps: any) => {
  let { object, schema } = ownProps;
  const tourney = state.tourney;

  return {
    tourney,
    object
  };
};

// The technology isn't there yet
// https://github.com/DefinitelyTyped/DefinitelyTyped/issues/9951
// @connect(mapStateToProps, mapDispatchToProps)
export class FormComponent extends React.Component<any, any> {
  public constructor(...args: any[]) {
    super(...args);
    const object = denormalize(this.props.object, this.props.schema, this.props.tourney) || {};
    this.state = {
      object
    };
  }

  public componentWillReceiveProps(nextProps: any) {
    // Keep our state object in sync w/ the prop
    if (!_.isUndefined(nextProps.object)) {
      this.setState({ object: nextProps.object });
    }
  }

  public handleChange(e: Event, props: any) {
    const target = e.target as any;
    const { object } = this.state;

    switch (props.type) {
      case 'checkbox':
        _.set(object, props.name, props.checked);
        break;
      case 'file':
        _.set(object, props.name, target.files[0]);
        break;
      default:
        _.set(object, props.name, props.value);
    }

    this.setState({ object: Object.assign({}, object) });

    if (this.props.onChange) {
      this.props.onChange(this.state.object);
    }
  }

  public toggleRaw(e: Event, props: any) {
    this.setState({ showRaw: props.checked });
  }

  public render() {
    return <SUI.Form>
      {this.formFields}
      <SUI.Form.Field>
        <SUI.Checkbox toggle label="Show raw data" onChange={this.toggleRaw.bind(this)} />
        <SUI.Segment style={{ backgroundColor: 'rgb(0, 43, 54)', display: this.state.showRaw ? 'block' : 'none' }}>
          <JSONTree data={this.state.object} />
        </SUI.Segment>
      </SUI.Form.Field>
      <SUI.Button positive loading={this.state.loading} onClick={this.submit.bind(this)}>Save</SUI.Button>
      <SUI.Button negative onClick={this.props.onCancel}>Cancel</SUI.Button>
    </SUI.Form>;
  }

  private insertAssociation(name: string, model: any) {
    const { object } = this.state;
    let value = _.get(object, name);
    let defaultValue = {} as any;

    model.properties.forEach((property: any) =>
                             defaultValue[property.name] = property.default)

    if (!_.isArray(value)) {
      value = [defaultValue];
    } else {
      value.push(defaultValue);
    }

    _.set(object, name, value);
    this.setState({ object: Object.assign({}, object) });
  }

  private get formFields() {
    return this.getFormFields(this.props.for, this.state.object);
  }

  private getFormFields(model: any, object: any, namespace: string = ''): JSX.Element[] {
    let fields: JSX.Element[] = [];

    model.properties.forEach((property: any, index: Number) => {
      let options = property.options;
      const propertyAliases = (property.aliasFor || []).concat(property.name);

      switch (property.type) {
        case 'socials':
          const AVAILABLE_SOCIALS = [
            // 'instagram',
            // 'reddit',
            // 'snapchat',
            // 'tumblr',
            'twitch',
            'twitter',
            'facebook',
            'youtube',
          ];

          AVAILABLE_SOCIALS.forEach((social: string) => {
            const fieldName = 'social['+social+']';
            const value = _.get(object, ['social', social]);
            fields.push(<FormField
              name={fieldName}
              key={fieldName}
              placeholder={social}
              type="text"
              value={value}
              errors={this.errorsFor(propertyAliases)}
              onChange={this.handleChange.bind(this)}
            />);
          });
          break;
        case 'draft':
          const onDraftChange = (editorState: EditorState) => {
            let object = { ...this.state.object };
            let contentState = editorState.getCurrentContent();
            let rawContentState = convertToRaw(contentState);
            this.setState({
              object: {
                ...object,
                [property.name]: JSON.stringify(rawContentState)
              }
            });
          };
          fields.push(
            <MediumEditor
              onUpdate={onDraftChange}
              initialState={object[property.name]}
              placeholder={property.placeholder}
            />
          );
          break;
        case 'association':
          let assocFields: JSX.Element[] = [];
          let propertyPath = namespace !== '' ? namespace + '[' + property.name + ']' : property.name;

          if (object[property.name]) {
            object[property.name].forEach((item: any, i: Number) => {
              assocFields = assocFields.concat(this.getFormFields(property.model, item, propertyPath + '['+i+']'));
            });
          }

          assocFields.push(
            <SUI.Button
              key={property.name}
              onClick={(e: React.SyntheticEvent<any>) => {
                e.preventDefault();
                this.insertAssociation(propertyPath, property.model);
              }}
            >Add {singularize(property.name)}</SUI.Button>
          );

          fields.push(
            <SUI.Form.Group className="grouped" key={propertyPath}><SUI.Segment>
            <SUI.Header as='h3' dividing>{property.name}</SUI.Header>
            {assocFields}
            </SUI.Segment></SUI.Form.Group>
          );

          break;
        default:
          if (!options && property.collection) {
            const items: Object = _.get(this.props, property.collection, {});
            options = _.map(items, (e: any) => {
              return {
                text: e.name || e.id,
                value: e.id
              };
            });
          }

          const fieldName = namespace !== '' ? namespace + '['+property.name+']' : property.name;
          let value = object[property.name];
          if (!value && property.default) {
            value = typeof property.default === 'function' ? property.default() : property.default;
          }

          fields.push(<FormField
            name={fieldName}
            key={fieldName}
            placeholder={property.placeholder || property.name}
            options={options}
            type={property.type}
            value={value}
            errors={this.errorsFor(propertyAliases)}
            onChange={this.handleChange.bind(this)}
          />);
      }
    });
    return fields;
  }

  private errorsFor(names: string[]): boolean | JSX.Element {
    let labels = [];
    for (const name of names) {
      if (this.state.object.errors && this.state.object.errors[name]) {
        labels.push(<SUI.Label basic color="red" pointing key={name}>
                    {this.state.object.errors[name]}
                    </SUI.Label>);
      }
    }
    return labels.length > 0
      ? <div className="errors">{labels}</div>
      : false;
  }

  private submit(e: any) {
    e.preventDefault();

    this.setState({ loading: true });
    const { object } = this.state;

    // serialize anything with a moment date
    _.forEach(object, (value, key) => {
      if (moment.isMoment(value)) {
        object[key] = value.toISOString();
      }
    });

    let action = this.props.action(this.state.object),
      promise = this.props.dispatch(action);

    promise
    .then((json: any) => {
      if (typeof this.props.onSuccess === 'function') {
        this.props.onSuccess(json);
      }

      return this.setState({
        loading: false,
        object: Object.assign(object, json.data)
      });
    }).catch((json: any) => {
      return this.setState({
        loading: false,
        object: Object.assign(object, json.data)
      });
    });
  }
}

export const Form = connect(mapStateToProps, mapDispatchToProps)(FormComponent);
Form.propTypes = {
  schema: React.PropTypes.any.isRequired
};

export * from './field';
