import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { connect } from 'react-redux';
import {
  denormalize,
  SeriesSchema,
  SeriesModel,
  MatchModel,
  createOrUpdateSeries,
  getTeams,
  getGames,
  getMatches,
  getSeries
} from 'tourney-sdk-react';

import { SeriesList } from './list';
import { Series } from './model';
import { Form } from '../form';

import { Container, Modal, Segment, Grid, Button, Menu } from 'semantic-ui-react';

class SeriesComponent extends React.Component<any, any> {
  public static mapStateToProps(state: any) {
    const seriesIds = Object.keys(state.tourney.series);
    const series = denormalize(seriesIds, [SeriesSchema], state.tourney);
    return {
      series
    }
  }

  public constructor(...args: any[]) {
    super(...args);

    this.state = {
      form: {
        series: {
          name: ''
        },
        visible: false
      },
      loading: true
    };
  }

  public componentDidMount() {
    const seriesChain = this.props.dispatch(getSeries())
      .then((series: SeriesModel[]) => {
        return this.props.dispatch(getMatches());
      });

    Promise.all([
      this.props.dispatch(getGames()),
      this.props.dispatch(getTeams()),
      seriesChain,
    ])
    .then((response: any) => this.setState({ loading: false }));
  }

  public editSeries(series: SeriesModel) {
    this.setState({ form: {
      series,
      visible: true
    }});
  }

  public showSeriesForm() {
    this.setState({ form: { visible: true } });
  }

  public hideSeriesForm() {
    this.setState({ form: { visible: false } });
  }

  public handleFormChange(series: any) {
    this.setState({ form: { series, visible: true } });
  }

  public render() {
    const listActions = {
      edit: this.editSeries.bind(this)
    };

    const listContent = this.state.loading
      ? undefined
      : <SeriesList
          series={this.props.series}
          actions={listActions}
        />;

    return <Container fluid>
      <Modal basic open={this.state.form.visible}>
        <Segment padded>
          <Form
            for={Series}
            schema={SeriesSchema}
            action={createOrUpdateSeries}
            object={this.state.form.series}
            onChange={this.handleFormChange.bind(this)}
            onSuccess={this.hideSeriesForm.bind(this)}
            onCancel={this.hideSeriesForm.bind(this)}
          />
        </Segment>
      </Modal>

      <Menu attached="top">
        <Menu.Item><Button onClick={this.showSeriesForm.bind(this)}>Create</Button></Menu.Item>
      </Menu>

      <Segment padded attached="bottom" loading={this.state.loading}>
        {listContent}
      </Segment>
    </Container>;
  }
}

export const SeriesIndex = connect(SeriesComponent.mapStateToProps)(SeriesComponent);
export * from './list';
