import * as qs from 'qs';
import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { connect } from 'react-redux';
import { isEqual } from 'lodash';
import {
  denormalize,
  TournamentEntrySchema,
  createOrUpdateTournamentEntry,
  getTournamentEntries
} from 'tourney-sdk-react';

import { TournamentEntryList } from './list';
import { TournamentEntry } from './model';
import { Form, FormField } from '../form';
import * as filter from '../../utils/filter';

import { Dropdown, Container, Modal, Segment, Grid, Button, Menu } from 'semantic-ui-react';

class TournamentEntriesComponent extends React.Component<any, any> {
  public static mapStateToProps(state: any, ownProps: any) {
    const cache = state.tourney.queries[filter.cacheKey()];
    let tournamentEntries;
    if (cache && cache.idMap) {
      const tournamentEntryIds = cache.idMap;
      tournamentEntries = denormalize(tournamentEntryIds, [TournamentEntrySchema], state.tourney);
    } else {
      const tournamentEntryIds = Object.keys(state.tourney.tournament_entries);
      tournamentEntries = denormalize(tournamentEntryIds, [TournamentEntrySchema], state.tourney);
    }
    return {
      tournamentEntries
    }
  }

  public constructor(...args: any[]) {
    super(...args);

    this.state = {
      form: {
        tournamentEntry: {
          team_id: undefined,
          tournament_id: undefined,
          flags: []
        }
      },
      filter: filter.get(),
      loading: true,
      formVisible: false
    };
  }

  public refresh() {
    const query = filter.get();
    const _query = Object.assign({
      cache: filter.cacheKey()
    }, query);
    this.props.dispatch(getTournamentEntries(_query)).
      then((response: any) => this.setState({ loading: false }));
  }

  public componentDidMount() {
    this.refresh();
  }

  public componentWillUpdate(nextProps: any) {
    if (!isEqual(this.props.location.search,
                 nextProps.location.search)) {
      this.refresh();
    }
  }

  public editTournamentEntry(tournamentEntry: any) {
    this.setState({ form: {
      tournamentEntry,
      visible: true
    }});
  }

  public showTournamentEntryForm() {
    this.setState({ form: { visible: true } });
  }

  public hideTournamentEntryForm() {
    this.setState({ form: { visible: false } });
  }

  public handleFormChange(tournamentEntry: TournamentEntry) {
    this.setState({ form: {
      tournamentEntry,
      visible: true
    }});
  }

  public filter(e: Event, props: any) {
    filter.update({[props.name]: props.value});
    this.setState({
      filter: filter.get()
    });
    this.refresh();
  }

  public render() {
    const flags = [{
      text: 'Payment Receive',
      value: 'payment_received'
    }, {
      text: 'Worst Team Ever',
      value: 'best_team_ever'
    }];

    const filterFlags = <FormField
      type="select-multiple"
      name="flags"
      value={this.state.filter.flags || []}
      placeholder="Filter by flag"
      options={flags}
      onChange={this.filter.bind(this)}
      errors={false}
    />;

    const listActions = {
      edit: this.editTournamentEntry.bind(this)
    };

    return <Container fluid>
      <Modal basic open={this.state.form.visible}>
        <Segment padded>
          <Form
            for={TournamentEntry}
            schema={TournamentEntrySchema}
            object={this.state.form.tournamentEntry}
            action={createOrUpdateTournamentEntry}
            onChange={this.handleFormChange.bind(this)}
            onSuccess={this.hideTournamentEntryForm.bind(this)}
            onCancel={this.hideTournamentEntryForm.bind(this)}
          />
        </Segment>
      </Modal>

      <Menu attached="top">
        <Menu.Item><Button onClick={this.showTournamentEntryForm.bind(this)}>Create</Button></Menu.Item>
        <Menu.Menu position="right">
          <Menu.Item>
          {filterFlags}
          </Menu.Item>
        </Menu.Menu>
      </Menu>

      <Segment padded attached="bottom" loading={this.state.loading}>
        <TournamentEntryList
          actions={listActions}
          tournamentEntries={this.props.tournamentEntries}
        />
      </Segment>
    </Container>;
  }
}

export const TournamentEntries = connect(TournamentEntriesComponent.mapStateToProps)(TournamentEntriesComponent);
export * from './list';
