class V1::BaseController < ApplicationController
  include Swagger::Blocks

  before_action :set_client!

  # Passport is special in that it accepts a token as User-ID and
  # Client#ident as Client-ID to avoid circular calls or more convoluted code
  # for #token and #client above.
  def passport
    @passport ||= PassportService.new(
      token: auth_token.to_s,
      client: Hashie::Mash.new(id: client_id.to_s),
      context: self
    )
  end

  def edb
    @edb ||= EdbService.new(
      user: token ? token.user : nil,
      client: client,
      context: self
    )
  end

  def self.cache_actions(*actions)
    opts = actions.extract_options!
    cache_opts = {}
    cache_opts[:ex] = opts[:expires_in].value if opts.include? :expires_in
    around_action only: actions do |controller, action|
      controller.cache_result(action, cache_opts)
    end
  end

  def cache_result(block, cache_opts)
    auth_token_hash = Digest::SHA1.hexdigest(auth_token.to_s)
    key = [request.fullpath, client_id, auth_token_hash].join('|')
    begin
      body = Rails.cache.fetch(key)
      if !body
        body = block.call
        Rails.cache.write(key, body, cache_opts) if response.status == 200
      end
    rescue SocketError, Redis::CannotConnectError
      body = block.call
    end
    self.response_body = body
    self.content_type = Mime[:json]
  end

  def self.cache_flush_actions(*actions)
    after_action :cache_flush, only: actions
  end

  def cache_flush()
    # If a connection to the cache drops before a request comes in that should clear it,
    # we should ideally queue up a flush for the next time connection to cache is re-established.
    Rails.cache.clear if response.status == 200 rescue nil
  end

  private

  def auth_token
    return @auth_token if defined?(@auth_token)
    stripped = request.headers['Authorization']&.gsub(/^Bearer/i,'')&.strip
    return @auth_token = stripped unless stripped.blank?
    @auth_token = params[:token]
  end

  def token
    return @token if @token
    return nil unless auth_token
    @token = passport.token(auth_token, raise_on_error: [500, 400]).dig(:json, :data)
    return false unless @token.is_a?(V1::TokenPresenter)
    @token = Hashie::Mash.new(@token.as_json)
  end

  attr_reader :client

  def client_id
    @client_id ||= params.delete(:client_id) || request.headers['Client-ID']
  end

  def set_client!
    raise ActionController::BadRequest.new('Client ID missing') unless client_id
    @client = Rails.cache.fetch(['client', client_id].join(':')) do
      passport.client(client_id, raise_on_error: true).dig(:json, :data).as_json
    end
    raise ActionController::BadRequest.new('Invalid Client ID.') if !@client
    @client = Hashie::Mash.new(@client)
  rescue => e
    render_exception(e)
    return false
  end

  # Swagger::Blocks aliases, shortcuts

  # I really ~don't like~ _hate_ the DSL of Swagger::Blocks, so here we are.
  def self.api_path(path, options = {})
    options.reverse_merge!(
      produces: ['application/json'],
      tags: []
    )

    options[:tags] += [
      self.name.split('::')[-1].gsub(/Controller/, '').underscore
    ]

    parameters = options.delete(:params) || {}
    parameters.reverse_merge!(
      'Client-ID' => {
        in: :header,
        type: :string,
        default: ENV['SWAGGER_CLIENT_ID']
      }
    )

    if options.delete(:requires_user)
      parameters['Authorization'] = {
        in: :header,
        type: :string,
        default: ENV['SWAGGER_AUTHORIZATION']
      }
    end

    responses = options.delete(:responses) || {}

    swagger_path path do
      operation options.delete(:method) || :get do
        for k, value in options do
          if value.is_a?(Hash)
            key k, *value
          else
            key k, value
          end
        end

        for param, keys in parameters do
          keys.reverse_merge!(
            in: :query,
            required: false,
            type: :string
          )

          parameter do
            key :name, param

            for k, value in keys do
              if value.is_a?(Hash)
                key k, *value
              else
                key k, value
              end
            end
          end
        end

        for status, keys in responses do
          schema_name = keys.delete(:schema)
          keys.reverse_merge!(
            description: schema_name || ''
          )

          response status do
            schema do
              property :status do
                key :type, :integer
                key :format, :int64
                key :default, 200
              end
              property :data do
                if schema_name && schema_name.is_a?(Array)
                  key :type, :array
                  items do
                    key :'$ref', schema_name[0]
                  end
                else
                  key :'$ref', schema_name
                end
              end
            end

            for k, value in keys do
              if value.is_a?(Hash)
                key k, *value
              else
                key k, value
              end
            end
          end
        end
      end
    end
  end
end
