class Service
  class ParameterMissing < StandardError; end

  class_attribute :config

  attr_reader :context

  self.config = Hashie::Mash.new(
    endpoint: '',
    protocol: 'http',
    version: '',
    params: {},
    headers: {
      'Accept' => 'application/json'
    },
    client: nil,
    user: nil,
    token: nil,
  )

  def self.inherited(klass)
    klass.config = config.dup
  end

  def self.endpoint(value)
    # Protocol should be omitted, we prepend the double slash for URI.parse
    value = "//#{value}" if value.scan('//').empty?
    value.gsub!(/[\/]*$/, '')
    config.endpoint = value
  end

  def endpoint
    config.endpoint
  end

  def endpoint=(value)
    self.class.endpoint(value)
  end

  def self.version(value)
    config.version = value
  end

  def initialize(config = {})
    @config = Hashie::Mash.new(self.config.dup.merge(config))
    @context = @config.context
    @config.headers['Client-ID'] = @config.client.id.to_s if @config.client
    @config.headers['Token'] = @config.token if @config.token
    if @config.user
      @config.headers['User-ID'] = @config.user.id
      if @config.client&.dig(:settings, :backend) && @config.user.roles
        @config.headers['X-Roles'] = @config.user.roles.join(',')
      end
    end
  end

  # Callers and connections

  attr_writer :conn

  def conn
    @conn ||= Faraday.new(url: _url) do |f|
      f.request :json
      f.request :url_encoded
      f.request :multipart
      f.adapter Faraday.default_adapter
    end
  end

  def request(method, path, params: {}, headers: {}, resource: nil, raise_on_error: false)
    params = prepare_params(params)
    headers = prepare_headers(headers, params)
    request_time = Time.now.to_f
    response = conn.send(method, _path(path), params, headers)
    log("#{'%.2f' % ((Time.now.to_f - request_time) * 1000.0)}ms #{response.status} #{method.to_s.upcase} #{_url(path)}: #{headers.inspect} #{params.inspect}")
    if (raise_on_error.is_a?(Array) && raise_on_error.include?(response.status)) ||
      (response.status != 200 && raise_on_error === true)
      # If something went wrong for a `raise_on_error: true` call,
      # all bets about the response format are off. Just try to extract something useful.
      begin
        json = JSON.parse(response.body)
        error = json['error'] || json['errors'] || 'unexpected response'
      rescue => e
        error = "received non-JSON reply for #{method.to_s.upcase} #{_url(path)}"
        Rollbar.error(e, error)
      end
      raise Edb::Exceptions::HttpError.new(response.status, error, {data: response.body})
    end
    Response.new(context: context, path: path, response: response, resource: resource)
  end

  def get(*args);    request(:get, *args); end
  def post(*args);   request(:post, *args); end
  def patch(*args);  request(:patch, *args); end
  def delete(*args); request(:delete, *args); end

  private

  def log(msg)
    puts %(#{Time.now.strftime('%H:%M:%S.%L')} [#{self.class.name}] #{msg})
  end

  def _url(path = '')
    "#{config.protocol}:#{config.endpoint}/#{_path(path) if path.present?}"
  end

  # If config.version is a string, we prepend it to the request path
  def _path(path)
    return path.gsub(/^\//, '') unless config.version.present? && path[0] != '/'
    [config.version, path].join('/')
  end

  def prepare_headers(headers, params)
    params = prepare_params(params)
    if params.values.collect(&:class).include?(UploadIO)
      headers['Content-Type'] = 'multipart/form-data'
    end
    config.headers.to_h.merge(headers.to_h)
  end

  def prepare_params(params)
    params = ActionController::Parameters.new(params).permit!
    [:action, :controller].each {|k| params.delete(k)}
    params.each do |key, value|
      if value.is_a?(ActionDispatch::Http::UploadedFile)
        params[key] = Faraday::UploadIO.new(params[key].to_io, params[key].content_type)
      end
    end
    config.params.to_h.merge(params.to_h)
  end
end
