require 'rails_helper'

describe V1::BaseController, type: :controller do
  describe '#auth_token' do
    it 'should strip the Authorization header' do
      controller.request.headers['Authorization'] = 'Bearer token'
      expect(controller.send(:auth_token)).to eq('token')
      controller.request.headers['Authorization'] = 'token'
      expect(controller.send(:auth_token)).to eq('token')
    end

    it 'should prefer the Authorization header over the token param' do
      controller.request.headers['Authorization'] = 'Bearer right'
      controller.params[:token] = 'wrong'
      token = controller.send(:auth_token)
      expect(token).to eq('right')
    end

    it 'should treat a present but empty Authorization header as if it was not provided' do
      ['', ' ', '    '].each do |content|
        controller.request.headers['Authorization'] = content
        expect(controller.send(:auth_token)).to be_nil
        expect(controller.send(:token)).to be_nil
      end
    end
  end

  describe '#token' do
    before :each do
      allow(controller).to receive_message_chain('passport.token').
        and_return({json: {data: V1::TokenPresenter.new({token: 'testing'})}})
    end

    it 'should return a Mash object when valid auth_token is supplied' do
      controller.request.headers['Authorization'] = 'Bearer testing'
      token = controller.send(:token)
      expect(token).to be_a(Hashie::Mash)
    end
  end

  describe 'Passport service' do
    it 'sends Token headers to Passport' do
      controller.request.headers['Authorization'] = 'Bearer testing'
      headers = controller.send(:passport).instance_variable_get('@config').headers
      expect(headers['Token']).to eq('testing')
      expect(headers).to_not include('User-ID')
    end
  end

  describe 'EDB service' do
    it 'sends User-ID headers to EDB' do
      allow(controller).to receive_message_chain('passport.token').
        and_return({json: {data: V1::TokenPresenter.new({user: {id: 'test user id'}})}})
      controller.request.headers['Authorization'] = 'Bearer testing'
      headers = controller.send(:edb).instance_variable_get('@config').headers
      expect(headers['User-ID']).to eq('test user id')
      expect(headers).to_not include('Token')
    end
  end
end
