require 'rails_helper'

describe V1::BaseController, type: :request do
  describe '#set_client!' do
    it 'should render an error if a Client ID was not provided' do
      get '/v1/teams'
      expect(response.status).to eq(400), ->{puts response.body.inspect}
      expect(json['error']).to eq('Client ID missing')
    end

    it 'should render 404 for a non-existent Client ID' do
      stub_request(:get, %r{/v1/clients/[^/]+$}).
        to_return(status: 404, body: {status: 404, error: 'no such client'}.to_json)
      get '/v1/teams', params: {client_id: 'does-not-exist'}
      expect(response.status).to eq(404), ->{puts response.body.inspect}
      expect(json['error']).to eq('no such client')
    end
  end

  describe 'controller cache' do
    stub_client_validation(self)

    before :all do
      skip "Can't test caching behavior when Rails.cache is nil" if Rails.cache.nil?
    end

    before :each do
      Rails.cache.clear
    end

    it 'should cache successful responses' do
      teams = FactoryGirl.build_list(:team, 5)
      stub_request(:get, %r{/v1/teams$}).and_return(body: teams.to_json)
      # teams is an array of Team objects; flatten to primitive values:
      teams = JSON.parse(teams.to_json)

      get '/v1/teams', params: {client_id: 'dev'}
      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to eq(teams)

      # Stub GET /v1/teams to return a different set of teams without invalidating cache.
      # Because cache entry was not invalidated, reply should be cached copy of the first reply.
      stub_request(:get, %r{/v1/teams$}).
        and_return(body: FactoryGirl.build_list(:team, 3).to_json)
      get '/v1/teams', params: {client_id: 'dev'}
      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to eq(teams)
    end

    it 'should not cache errors' do
      stub_request(:get, %r{/v1/teams$}).
        and_return(status: 500, body: {status: 500, error: 'construct additional pylons'}.to_json)
      get '/v1/teams', params: {client_id: 'dev'}
      expect(response.status).to eq(500)

      # Update stub to return a successful reply.
      stub_request(:get, %r{/v1/teams$}).and_return(body: '[]')
      get '/v1/teams', params: {client_id: 'dev'}
      expect(response.status).to eq(200)
    end

    it 'should flush cache on any successful write/update/delete' do
      stub_request(:patch, %r{/v1/teams/foo}).
        and_return(body: FactoryGirl.build(:team).to_json)
      expect(Rails.cache).to receive(:clear).once.and_call_original
      patch '/v1/teams/foo', params: {client_id: 'dev'}
      expect(response.status).to eq(200)
    end

    it 'should not flush cache on an unsuccessful write/update/delete' do
      stub_request(:patch, %r{/v1/teams/foo}).and_return(status: 404)
      expect(Rails.cache).to_not receive(:clear)
      patch '/v1/teams/foo', params: {client_id: 'dev'}
      expect(response.status).to eq(404)
    end
  end
end
