require 'rails_helper'

describe V1::PassportController, type: :controller do
  stub_client_validation(self)

  describe 'GET /users/auth' do
    it 'should respond with status 200, but include status 302/redirect when initiating oauth cycle' do
      # /v1/users/auth calls passport/auth/:provider
      stub_request(:get, %r{/auth/twitch.*}).
        to_return(status: 302, body: 'Redirecting to whatever',
                  headers: {location: 'http://localhost/lolwut?state=teststate'})

      get :auth, params: {client_id: 'dev'}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['status']).to eq(302)
      expect(json).to include('token')
      expect(json).to include('redirect')
    end
  end

  describe 'GET /users/auth/callback/:provider' do
    it 'should render a JSON using a UserPresenter' do
      user = FactoryGirl.build(:user)
      stub_request(:get, %r{/auth/twitch/callback.*}).to_return(body: user.to_json)
      expect(V1::UserPresenter).to receive(:new).and_call_original
      # UserPresenter calls edb/person/:user_id
      person = FactoryGirl.build(:person, user_id: user.id)
      stub_request(:get, %r{/v1/people/#{user.id}$}).to_return(body: person.to_json)
      expect(V1::PersonPresenter).to receive(:new).and_call_original

      get :callback, params: {client_id: 'dev', provider: 'twitch'}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to be_a(Hash)
    end
  end
end
