require 'rails_helper'

describe V1::ScheduleController do
  stub_client_validation(self)

  before :each do
    Rails.cache.clear
  end

  it 'combines series and events' do
    events = (-3..3).map {|i| {'id' => "event-#{i+4}", 'starts_at' => i.days.from_now}}
    series = (-3..3).map {|i| {'id' => "series-#{i+4}", 'scheduled_at' => i.days.from_now + 12.hours}}
    stub_request(:get, %r{/v1/events.*}).to_return(body: events.to_json)
    stub_request(:get, %r{/v1/series.*}).to_return(body: series.to_json)

    get '/v1/schedule', params: {client_id: 'dev'}
    expect(response.status).to eq(200), ->{ pp json }

    expected_ids = (1..7).map {|i| ["event-#{i}", "series-#{i}"]}.flatten
    expect(json_ids).to eq(expected_ids)
    expect(json['data'].map {|obj| obj['type']}).to eq(['Event', 'Series'].cycle(7).to_a)
  end

  it "doesn't replace Event or Series types already in the results" do
    event = {id: 'event-1', type: 'Event::Broadcast', starts_at: Time.now}
    series = {id: 'series-1', type: 'Series::Bo5', scheduled_at: Time.now + 1.hour}
    stub_request(:get, %r{/v1/events.*}).to_return(body: [event].to_json)
    stub_request(:get, %r{/v1/series.*}).to_return(body: [series].to_json)

    get '/v1/schedule', params: {client_id: 'dev'}
    expect(response.status).to eq(200), ->{ pp json }
    expect(json_ids).to eq([event[:id], series[:id]])
    expect(json['data'].map {|obj| obj['type']}).to eq([event[:type], series[:type]])
  end

  it "filters out Events and Series with no start/scheduled time" do
    events = 10.times.map {|i| {'id' => "event-#{i+1}", 'starts_at' => (i+1) % 2 == 0 ? nil : i.days.from_now}}
    series = 10.times.map {|i| {'id' => "series-#{i+1}", 'scheduled_at' => (i+1) % 2 == 0 ? nil : i.days.from_now}}
    stub_request(:get, %r{/v1/events.*}).to_return(body: events.to_json)
    stub_request(:get, %r{/v1/series.*}).to_return(body: series.to_json)

    get '/v1/schedule', params: {client_id: 'dev'}
    expect(response.status).to eq(200), ->{ pp json }
    expect(json['data'].length).to eq(10), ->{ pp json }
    exp_ids = (1..10).step(2).map {|i| ["event-#{i}", "series-#{i}"]}.flatten
    expect(json_ids.sort).to eq(exp_ids.sort)
  end

  it 'applies limit and offset' do
    events = 10.times.map {|i| {'id' => "event-#{i+1}", 'starts_at' => i.days.from_now}}
    stub_request(:get, %r{/v1/events.*}).to_return(body: events.to_json)
    stub_request(:get, %r{/v1/series.*}).to_return(body: '[]')

    get '/v1/schedule', params: {client_id: 'dev', limit: '3', offset: '4'}
    expect(response.status).to eq(200), ->{ pp json }
    expect(json['data'].length).to eq(3), ->{ pp json }
    exp_ids = (5...8).map {|i| "event-#{i}"}
    expect(json_ids).to eq(exp_ids)
  end
end
