require 'rails_helper'

# TODO this is the request spec, marked as a controller spec, with most of the
# specs being request specs. Split this correctly ..

describe V1::UsersController, type: :controller do
  stub_client_validation(self)

  before :all do
    @users = FactoryGirl.build_list(:user, 5)
    @people = @users.map {|u| FactoryGirl.build(:person, user_id: u.id)}
    @user = @users[0]
    @person = @people[0]
  end

  before :each do
    # UserPresenter makes calls to edb/people/:id
    @people.each do |p|
      stub_request(:get, %r{/v1/people/#{p.user_id}}).to_return(body: p.to_json)
    end

    @users.each do |p|
      stub_request(:get, %r{/v1/users/#{p.id}}).to_return(body: p.to_json)
    end
  end

  def verify_user_embed(u)
    expect(u).to_not include('user_id')
    expect(u).to include('first_name')
  end

  context 'GET /users' do
    it 'should render a JSON using an array of UserPresenters' do
      stub_request(:get, %r{/v1/users$}).to_return(body: @users.to_json)
      expect(V1::UserPresenter).to receive(:new).and_call_original.exactly(@users.length).times
      expect(V1::PersonPresenter).to receive(:new).and_call_original.exactly(@users.length).times

      get :index, params: {client_id: 'dev'}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to be_a(Array)
      json['data'].each {|u| verify_user_embed(u)}
    end
  end

  context 'GET /users/:id' do
    it 'should render a JSON using a UserPresenter' do
      expect(V1::UserPresenter).to receive(:new).and_call_original
      expect(V1::PersonPresenter).to receive(:new).twice.and_call_original

      get :show, params: {client_id: 'dev', id: @user.id}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to be_a(Hash)
      verify_user_embed(json['data'])
    end

    it "should render the person, even if the user doesn't exist" do
      stub_request(:get, %r{/v1/users/#{@person.user_id}}).to_return(body: '{}', status: 404)
      get :show, params: {client_id: 'dev', id: @person.user_id}
      pp response.body
      pp response.status
      verify_user_embed(json['data'])
    end
  end

  context 'PATCH /users/:id' do
    it 'should call edb.update_person before passport.update_user' do
      stub_request(:patch, %r{/v1/users/[^/]+$}).to_return(body: @user.to_json)
      stub_request(:patch, %r{/v1/people/[^/]+$}).to_return(body: @person.to_json)

      expect(controller.edb).to receive(:update_person)
      expect(controller.passport).to receive(:update_user)

      patch :update, params: {client_id: 'dev', id: @person.user_id, first_name: 'Markus'}
    end

    it "should render the person, even if the user doesn't exist" do
      stub_request(:patch, %r{/v1/users/[^/]+$}).to_return(body: '{}', status: 404)
      stub_request(:patch, %r{/v1/people/[^/]+$}).to_return(body: @person.to_json)
      patch :update, params: {client_id: 'dev', id: @person.user_id, first_name: 'Markus'}
      verify_user_embed(json)
    end

    it 'should render a JSON using a UserPresenter' do
      stub_request(:patch, %r{/v1/users/[^/]+$}).to_return(body: @user.to_json)
      stub_request(:patch, %r{/v1/people/[^/]+$}).to_return(body: @person.to_json)

      expect(V1::UserPresenter).to receive(:new).and_call_original
      # Called twice, because we're first patching the user, then requesting it
      # from the UserPresenter
      expect(V1::PersonPresenter).to receive(:new).twice.and_call_original

      patch :update, params: {client_id: 'dev', id: @user.id}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to be_a(Hash)
      verify_user_embed(json['data'])
    end
  end

  context 'GET /users/verify/:token' do
    it 'should render a JSON using a UserPresenter' do
      stub_request(:get, %r{/v1/users/[^/]+$}).to_return(body: @user.to_json)
      # this call also sends a user_event to EDB
      stub_request(:post, %r{/v1/user_events$}).to_return(status: 200)
      expect(V1::UserPresenter).to receive(:new).and_call_original
      expect(V1::PersonPresenter).to receive(:new).and_call_original

      get :verify_email, params: {client_id: 'dev', email_token: 'foo'}

      expect(response.status).to eq(200), "got error reply: #{response.body.inspect}"
      expect(json['data']).to be_a(Hash)
      verify_user_embed(json['data'])
    end
  end

  describe 'PATCH /users/:id' do
    it 'should rewind any IO objects in parameters' do
      user = FactoryGirl.build(:user)
      stub_request(:get, %r{/v1/users/#{user.id}}).to_return(body: user.to_json)
      person = FactoryGirl.build(:person, user_id: user.id)
      stub_request(:get, %r{/v1/people/#{user.id}}).to_return(body: person.to_json)

      path = Tempfile.open('') {|f| f.write("foo bar baz"); f.path}
      file = Rack::Test::UploadedFile.new(path)

      [['users',user], ['people',person]].each do |pair|
        endpoint, obj = pair
        stub_request(:patch, %r{/v1/#{endpoint}/#{user.id}}).
          with do |request|
            # How NOT to write a multipart parser library, courtesy of Rack:
            Rack::Multipart::Parser.parse(
              StringIO.new(request.body),                 # the only arg actually needed
              request.headers['Content-Length'].to_i,     # length of body
              request.headers['Content-Type'],            # only used to get boundary; first line of body
              Rack::Multipart::Parser::TEMPFILE_FACTORY,  # should have been defined with default
              Rack::Multipart::Parser::BUFSIZE,           # should have been defined with default
              Rack::Utils.default_query_parser,           # should have been defined with default
            ).params['io'][:tempfile].read == "foo bar baz"
          end.to_return(body: obj.to_json)
      end

      patch :update, params: {client_id: 'dev', id: user.id, io: file}
      expect(response.status).to eq(200), ->{puts response.body}
    end
  end
end
