require 'rails_helper'

class TestService < Service
end

shared_examples "service" do |service_class|
  let(:service) { service_class.new }

  it 'configures an endpoint' do
    expect(service.endpoint).to_not be_blank
  end

  it 'configures a valid URI endpoint' do
    uri = URI.parse(service.endpoint)
    expect(uri.host).to_not be_blank
  end
end

describe Service do
  context 'config' do
    # Make sure we're instantiating a new config object upon inheritance
    # so we're not writing to the referenced Service.config defaults
    it 'should dup config on inherit' do
      TestService.endpoint 'example.com'
      expect(Service.config.endpoint).to_not match(/example.com/)
    end

    it 'should dup config on initialize' do
      svc = Service.new
      svc2 = Service.new
      svc.config.endpoint = 'example.com'
      expect(Service.config.endpoint).to_not match(/example.com/)
      expect(svc2.config.endpoint).to_not match(/example.com/)
    end

    describe 'endpoint' do
      before :each do
        TestService.endpoint('example.com:80')
      end

      it 'strips trailing slash' do
        expect { TestService.endpoint('localhost:80/') }.to \
          change { TestService.config.endpoint }
          .from(TestService.config.endpoint).to('//localhost:80')
      end

      it 'prepends missing double slash when protocol is omitted' do
        expect { TestService.endpoint('localhost:80') }.to \
          change { TestService.config.endpoint }
          .from(TestService.config.endpoint).to('//localhost:80')
      end
    end
  end

  describe 'connection' do
    before :all do
      @service = TestService.new(endpoint: '//localhost')
    end

    [:get, :post].each do |method|
      describe "##{method}" do
        it "should execute #{method} requests" do
          @service.config.version = ''
          body = {version_set: false}.to_json
          stub_request(method, %r{localhost/teams$}).to_return(body: body)
          response = @service.send(method, 'teams')
          expect(response.env.method).to eq(method)
          expect(response.body).to eq(body)
        end

        it 'should prepend version string to path if set' do
          @service.config.version = 'v3'
          body = {version_set: true}.to_json
          stub_request(method, %r{localhost/v3/teams$}).to_return(body: body)
          response = @service.send(method, 'teams')
          expect(response.env.method).to eq(method)
          expect(response.body).to eq(body)
        end
      end
    end

    it 'should not prepend version string if path begins with a /' do
      @service.config.version = 'v4'
      body = {name: 'lolwut'}.to_json
      stub_request(:get, %r{localhost/teams$}).to_return(body: body)
      response = @service.get('/teams')
      expect(response.body).to eq(body)
    end
  end

  describe '#request' do
    before :all do
      @service = TestService.new(endpoint: '//localhost')
    end

    [:get, :post, :patch, :delete].each do |method|
      it "##{method.to_s} should raise on non-200 if requested" do
        exp_body = {error: "I'm a teapot"}.to_json
        stub_request(method, %r{localhost/teams$}).to_return(status: 418, body: exp_body)
        expect { @service.send(method, 'teams', raise_on_error: true) }.to raise_error do |error|
          expect(error).to be_a(Edb::Exceptions::HttpError)
          expect(error.status).to eq(418)
          expect(error.message).to eq("I'm a teapot")
          expect(error.extra[:data]).to eq(exp_body)
        end
      end
    end

    it 'should catch non-JSON nicely when asked to raise on error' do
      body = '<html><body><h1>Internal Server Error</h1><body></html>'
      stub_request(:get, /foo$/).to_return(status: 500, body: body)
      expect { @service.request(:get, 'foo', raise_on_error: true) }.to raise_error do |error|
        expect(error).to be_a(Edb::Exceptions::HttpError)
        expect(error.status).to eq(500)
        expect(error.message).to eq("received non-JSON reply for GET http://localhost/foo")
        expect(error.extra[:data]).to eq(body)
      end
    end
  end

  describe 'initialize' do
    it 'should set Client-ID header if initialized with a client' do
      service = TestService.new(client: Hashie::Mash.new(id: 'a client id'))
      headers = service.instance_variable_get('@config').headers
      expect(headers['Client-ID']).to eq('a client id')
      expect(headers).to_not include('User-ID')
      expect(headers).to_not include('Token')
    end

    it 'should set User-ID header if initialized with a user' do
      service = TestService.new(user: Hashie::Mash.new(id: 'a user id', roles: nil))
      headers = service.instance_variable_get('@config').headers
      expect(headers['User-ID']).to eq('a user id')
      expect(headers).to_not include('Client-ID')
      expect(headers).to_not include('Token')
    end

    it 'should set Token header if initialized with a token' do
      service = TestService.new(token: 'a secure token')
      headers = service.instance_variable_get('@config').headers
      expect(headers['Token']).to eq('a secure token')
      expect(headers).to_not include('Client-ID')
      expect(headers).to_not include('User-ID')
    end

    it 'should not set Admin header if user is absent' do
      service = TestService.new()
      expect(service.instance_variable_get('@config').headers['Admin']).to eq(nil)
    end

    it 'should not set Admin header if user is not admin' do
      service = TestService.new(user: {roles: ['foo','bar'], email: 'foo@example.com'})
      expect(service.instance_variable_get('@config').headers['Admin']).to eq(nil)
    end

    it 'should set X-Roles header for backend clients if user has roles' do
      service_params = {user: {roles: ['foo','bar'], email: 'foo@example.com'}}
      service = TestService.new(service_params)
      expect(service.instance_variable_get('@config').headers).to_not include('X-Roles')

      service_params[:client] = {settings: {backend: true}}
      service = TestService.new(service_params)
      expect(service.instance_variable_get('@config').headers['X-Roles']).to eq('foo,bar')
    end
  end

  describe 'prepare_params' do
    it 'should permit all parameters in ActionController::Parameters' do
      service = TestService.new
      ac_params = ActionController::Parameters.new({ test: 1 })
      expect(service.send(:prepare_params, ac_params)).to include({'test' => 1})
    end

    it 'should rewrite UploadedFile to Faraday::UploadIO' do
      service = TestService.new
      params = { test: ActionDispatch::Http::UploadedFile.new(tempfile: Tempfile.new) }
      _params = service.send(:prepare_params, params)
      expect(_params['test']).to be_a(Faraday::UploadIO)
    end
  end

  describe 'prepare_headers' do
    it 'should set Content-Type to multipart/form-data when params includes a file' do
      service = TestService.new
      params = { test: ActionDispatch::Http::UploadedFile.new(tempfile: Tempfile.new) }
      headers = service.send(:prepare_headers, {}, params)
      expect(headers['Content-Type']).to eq('multipart/form-data')
    end
  end

  describe 'Response' do
    it 'should have keys "status" and "data" on success' do
      base_response = double(status: 200, body: '{"foo": "bar"}')
      response = Service::Response.new(resource: 'base', response: base_response)
      response = JSON.parse(response.to_json)

      expect(response['status']).to eq(200)
      expect(response['json']['status']).to eq(200)
      expect(response['json']['data']).to eq({'foo' => 'bar'})
      expect(response['json']).to_not include('error', 'errors')
    end

    it "shouldn't break when given an empty response body" do
      base_response = double(status: 200, body: '')
      response = Service::Response.new(resource: 'base', response: base_response)
      response = JSON.parse(response.to_json)

      expect(response['status']).to eq(200)
      expect(response['json']['status']).to eq(200)
      expect(response['json']['data']).to eq({})
      expect(response['json']).to_not include('error', 'errors')
    end

    it 'should always have a Presenter or an array of Presenters in "data" for successful responses' do
      context = 'this is a context, lol'

      single_response = double(status: 200, body: '{"foo": "bar"}')
      response = Service::Response.new(resource: 'base', response: single_response, context: context)
      expect(response.instance_variable_get(:@context)).to eq(context)
      data = response.json[:data]
      expect(data).to be_a(V1::BasePresenter)
      expect(data.instance_variable_get(:@context)).to eq(context)

      array_response = double(status: 200, body: '[{"foo": "bar"}, {"baz": "qux"}]')
      response = Service::Response.new(resource: 'base', response: array_response, context: context)
      expect(response.instance_variable_get(:@context)).to eq(context)
      data = response.json[:data]
      expect(data).to be_a(Array)
      data.each do |obj|
        expect(obj).to be_a(V1::BasePresenter), ->{pp data}
        expect(obj.instance_variable_get(:@context)).to eq(context)
      end
    end
  end
end
