variable "service_name" { default = "tourney-edge" }
variable "service_env" { default = "production" }

provider "aws" {
  region = "us-west-2"
  profile = "twitch-esports-aws"
}

provider "aws" {
  alias = "dev"
  region = "us-west-2"
  profile = "twitch-esports-dev"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-aws/"
}

data "terraform_remote_state" "aws" {
  backend = "s3"
  config {
    bucket = "twitch-esports-aws"
    key = "tfstate/esports/terraform/aws/raw"
    profile = "twitch-esports-aws"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "tourney-edge" {
  name = "tourney-edge"
}

data "template_file" "edge_task_definition" {
  template = "${file("task-definitions/tourney-edge.json")}"
  vars {
    name = "tourney-edge"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.tourney-edge.repository_url, "https://", "")}:latest"
    cache_url = "redis://${aws_elasticache_cluster.default.cache_nodes.0.address}:6379/0"
    container_port = 3000
    cpu = 512
    mem = 512
  }
}

resource "aws_ecs_task_definition" "edge_task" {
  family = "tourney-edge"
  container_definitions = "${data.template_file.edge_task_definition.rendered}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-19"
  name = "tourney-edge"
  prefix = "tourney_edge"
  container_port = "3000"
  alb = true
  rails_json = true
  cluster = "tourney"
  env = "${var.service_env}"
  desired_count = 2
  # rlcs.gg
  certificate_arn = "arn:aws:acm:us-west-2:954268612818:certificate/bc6ef236-5ba0-4fd8-abe5-1eb774bdb559"
  task_definition = "${aws_ecs_task_definition.edge_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.aws.alarm_sns_topic_arn}"
}

resource "aws_route53_record" "production" {
  zone_id = "Z39V61UJPEWKZR" # esports.army
  name = "edge.tourney"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_route53_record" "internal" {
  zone_id = "ZALZ14ARD67SH" # esports.internal.justin.tv
  provider = "aws.dev"
  name = "${var.service_name}.${var.service_env}"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_route53_record" "ahgl" {
  zone_id = "Z3PPMYSIRHZ4EI" # ahgl.tv
  name = "api"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_route53_record" "rlcs" {
  zone_id = "Z140IZPCS2HRJW" # rlcs.gg
  name = "api"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

output "ecr_url" {
  value = "${aws_ecr_repository.tourney-edge.repository_url}"
}
