variable "service_name" { default = "tourney-edge" }
variable "service_env" { default = "staging" }

provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

provider "aws" {
  alias = "prod"
  region = "us-west-2"
  profile = "twitch-esports-aws"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-dev/"
}

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "tourney-edge" {
  name = "tourney-edge"
}

data "template_file" "edge_task_definition" {
  template = "${file("task-definitions/tourney-edge.json")}"
  vars {
    name = "tourney-edge"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.tourney-edge.repository_url, "https://", "")}:latest"
    cache_url = "redis://${aws_elasticache_cluster.default.cache_nodes.0.address}:6379/0"
    container_port = 3000
    cpu = 512
    mem = 512
  }
}

resource "aws_ecs_task_definition" "edge_task" {
  family = "tourney-edge"
  container_definitions = "${data.template_file.edge_task_definition.rendered}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-16"
  name = "tourney-edge"
  prefix = "tourney_edge"
  container_port = "3000"
  alb = true
  rails_json = true
  cluster = "tourney"
  env = "staging"
  desired_count = 2
  # staging.rlcs.gg us-west-2
  certificate_arn = "arn:aws:acm:us-west-2:157383839067:certificate/1dc70f4b-e723-49fd-bb88-c5f5faba9a36"
  task_definition = "${aws_ecs_task_definition.edge_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.dev.alarm_sns_topic_arn}"
}

resource "aws_route53_record" "rlcs" {
  provider = "aws.prod"
  zone_id = "Z140IZPCS2HRJW" # rlcs.gg
  name = "api.staging"
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

output "ecr_url" {
  value = "${aws_ecr_repository.tourney-edge.repository_url}"
}
