import * as _ from 'lodash';
import { denormalize } from 'normalizr';
import * as React from 'react';
import { connect } from 'react-redux';
import { bindActionCreators, Dispatch } from 'redux';
import {
  getTournaments,
  UserSchema,
  UserModel,
  updateUser,
  logoutUser,
} from 'tourney-sdk-react';

import { AppModel } from 'state/model';
import { NavigationContainer } from 'ui/components/navigation';
import { Footer } from 'ui/components/footer';
import { SignupIntercept } from 'ui/components/signup-intercept';

interface AppProps {
  children: JSX.Element;
  location: Location;
  getTournaments: Function;
}

interface PropsFromState {
  loggedIn: boolean;
  user: UserModel;
  requests: any[];
}

function mapStateToProps(state: AppModel): PropsFromState {
  const current = state.tourney.users.current;
  const loggedIn = current !== undefined;
  const user = loggedIn ? denormalize(current, UserSchema, state.tourney) : undefined;
  return {
    loggedIn,
    user,
    requests: state.tourney.requests,
  };
}

function mapDispatchToProps(dispatch: Dispatch<AppModel>) {
  return bindActionCreators({ logoutUser, updateUser, getTournaments }, dispatch);
}

class AppComponent extends React.Component<any, any> {
  constructor(...args: any[]) {
    super(args[0]);
    this.state = {
      showPlaceholders: false,
      showDebug: true
    };
  }

  public toggle(setting: string) {
    return (e: Event): any => {
      this.setState({
        [setting]: !this.state[setting]
      });
    };
  }

  public render() {
    const navigationProps: any = {
      location: this.props.location
    };

    const classes = ['app-container',
      this.state.showPlaceholders ? 'show-placeholders' : '',
      this.state.showDebug ? 'show-debug' : ''];

    return <div className={classes.join(' ')}>
      <div id="debug">
        <a onClick={this.toggle('showPlaceholders').bind(this)}>
          {this.state.showPlaceholders ? 'Hide' : 'Show'} Placeholders
        </a>
        <a onClick={this.toggle('showDebug').bind(this)}>
          {this.state.showDebug ? 'Hide' : 'Show'} Debug
        </a>
        <pre>{JSON.stringify(this.props.requests)}</pre>
      </div>
      <NavigationContainer {...navigationProps}/>
      {this.props.children}
      <Footer />
      {this.renderIntercept()}
    </div>;
  }

  private renderIntercept() {
    const { user, loggedIn, updateUser, logoutUser } = this.props;

    if (loggedIn && (_.isEmpty(user.settings) || !user.settings.tos_accepted)) {
      const props = {
        user,
        updateUser,
        logoutUser,
      };
      return <SignupIntercept {...props} />;
    }
    return null;
  }
}

export const App = connect(mapStateToProps, mapDispatchToProps)(AppComponent) as React.ComponentClass<AppProps>;
