import * as React from 'react';
const styles = require('./styles.scss');

export type ButtonSize = 'small' | 'medium' | 'large';
const BUTTON_SIZES: {[B in ButtonSize]: ButtonSize} = {
  small: 'small',
  medium: 'medium',
  large: 'large',
};

export type ButtonConnect = 'left' | 'right' | 'both' | 'top';
const BUTTON_CONNECTS: {[B in ButtonConnect]: ButtonConnect} = {
  left: 'left',
  right: 'right',
  both: 'both',
  top: 'top',
};

interface ButtonProps {
  size?: ButtonSize;
  connect?: ButtonConnect;
  primary?: boolean;
  secondary?: boolean;
  twoLine?: boolean;
  loading?: boolean;
  error?: boolean;
  hidden?: boolean;
  disabled?: boolean;
  onClick?: Function;
}

export class ButtonComponent extends React.Component<ButtonProps, void> {
  public static SIZES = BUTTON_SIZES;
  public static CONNECTS = BUTTON_CONNECTS;

  public render() {
    let classes = ['ui', 'button'];
    let buttonProps: {[key: string]: any} = {};

    // option based prop-value className
    classes = ['size', 'connect'].reduce((memo, propName) => {
      return this.checkClassOption(memo, propName);
    }, classes);

    // simple props with matching classNames check
    classes = ['loading', 'error', 'primary', 'secondary', 'hidden', 'disabled'].reduce((memo, propName) => {
      return this.checkClass(memo, propName);
    }, classes);

    // prop flags that use a different className
    classes = this.checkClass(classes, 'twoLine', 'two-line');

    // props
    buttonProps = this.checkProp(buttonProps, 'onClick');

    return (
      <button className={classes.join(' ')} {...buttonProps}>
        {this.props.children}
      </button>
      );
  }

  // check for class driven props
  private checkClass(classes: string[], propName: string, className?: string): string[] {
    if ((this.props as any)[propName]) {
      const addClass = className ? className : propName;
      return [...classes, addClass];
    }
    return [...classes];
  }

  // check for class driven props with options
  private checkClassOption(classes: string[], propName: string): string[] {
    if ((this.props as any)[propName]) {
      const propValue = (this.props as any)[propName];
      return [...classes, `${propName}-${propValue}`];
    }
    return [...classes];
  }

  // check prop
  private checkProp(buttonProps: {[key: string]: any}, propName: string): {[key: string]: any} {
    if ((this.props as any)[propName]) {
      return {
        ...buttonProps,
        [propName]: (this.props as any)[propName]
      };
    }
    return { ...buttonProps };
  }
}
