import * as _ from 'lodash';
import * as React from 'react';
import { Link } from 'react-router';
import { UserModel } from 'tourney-sdk-react';

import { Tournament } from 'common/constants';
import { AppModel } from 'state/model';
import { Button } from 'ui/components/button';
import { Tooltip } from 'ui/components/tooltip';

const styles = require('./styles/styles.scss');

const contentStates = {
  unverified: {
    title: 'One last Step',
    text: 'Verify your company email account before you can compete',
    icon: 'fa-bell'
  },
  intermediate: {
    title: 'Check your Inbox',
    text: 'We sent a verification email to your new address',
    icon: 'fa-warning'
  },
  verified: {
    title: 'You\'re good to go!',
    text: 'You are verified',
    icon: 'fa-check'
  }
};

export interface EmailVerificationProps {
  user: UserModel;
  loading: boolean;
}

export class EmailVerificationComponent extends React.Component<any, any> {
  public constructor(props: any) {
    super(props);
    const { user } = this.props;
    this.state = {
      email: user.is_verified ? user.email : user.email_new,
      loading: false,
      error: undefined,
    };
  }

  public handleChange(event: any) {
    this.setState({
      [event.target.name]: event.target.value,
      error: undefined,
    });
  }

  public get verificationState() {
    if (this.props.user.email_new !== null) {
      return 'intermediate';
    } else if (this.props.user.is_verified) {
      return 'verified';
    } else {
      return 'unverified';
    }
  }

  public get isLoading() {
    return this.props.loading || this.state.loading;
  }

  public render() {
    const { user } = this.props;
    let content = contentStates[this.verificationState];
    const verifiedClasses = ['email-verify'];
    verifiedClasses.push(this.verificationState);

    const updateUser = () => {
      this.setState({ loading: true });
      this.props.updateUser(_.assign(
        user,
        { email: this.state.email }
        )
      ).then(() => {
        this.setState({ loading: false });
      }).catch((err: any) => {
        this.setState({
          error: err.data.errors.email[0], // sketchy error search
          loading: false,
        });
      });
    };

    // email_token should not be present on production, we include it in
    // development and staging for convenience and testing.
    let emailVerificationLink;
    if (user.email_token) {
      const verificationUri = '/settings/verify/' + user.email_token;
      emailVerificationLink = <div style={{ background: 'lightblue', padding: '1rem' }}>
        Verification Link: <Link to={verificationUri}>{verificationUri}</Link>
      </div>;
    }

    const buttonProps = {
      loading: this.isLoading,
      connect: Button.CONNECTS.left,
      onClick: updateUser,
    };

    let tooltip;
    if (this.state.error) {
      tooltip = <Tooltip position="top-right">{this.state.error}</Tooltip>;
    }

    return (
      <div className={verifiedClasses.join(' ')}>
        <div className="left-content">
          <div className="icon">
            <i className={['fa', content.icon].join(' ')} />
          </div>
          <div className="content">
            <div className="title">{content.title}</div>
            <div className="text">{content.text}</div>
          </div>
        </div>
        {emailVerificationLink}
        {tooltip}
        <div className="email-container">
          <input type="text" className="email-form" placeholder="Your Company Email" autoComplete="off" autoCorrect="off" autoCapitalize="off" spellCheck={false} name="email" value={this.state.email} onChange={this.handleChange.bind(this)} />
          <Button {...buttonProps}><i className="fa fa-arrow-right"/></Button>
        </div>
      </div>
    );
  }
}
