import * as React from 'react';

import { Image } from '../image';
import { Tooltip } from 'ui/components/tooltip';

const styles = require('./styles.scss');

interface ImageUploadProps {
  src: string;
  onChange: Function;
  className?: string;
  defaultSrc?: string;
}

export class ImageUpload extends React.Component<any, any> {
  public constructor() {
    super();
    this.state = {
      state: '',
      error: undefined,
      previewSrc: null,
    };
  }

  public onChange(e: any) {
    const promise = this.props.onChange(e);
    if (promise instanceof Promise) {
      this.previewImage(e.target);
      this.setState({ state: 'loading' });
      promise
        .then(() => {
          this.setState({
            state: 'success'
          });
          // clear out preview src after animation
          setTimeout(() => {
            this.setState({ previewSrc: null });
          }, 2000);
        })
        .catch((err) => {
          this.setState({
            state: 'error',
            error: err.data.errors.logo[0],
          });
          this.resetError();
        });
    }
  }

  public editLogo() {
    this.logoInput.click();
  }

  public render () {
    let classes = ['image-upload', this.state.state, this.props.className];
    let currentClasses = ['current-container', this.state.state];
    let previewClasses = ['preview-container', this.state.state];

    let tooltip;
    if (this.state.state === 'error') {
      tooltip = <Tooltip>{this.state.error}</Tooltip>;
    }

    let logoSrc = this.props.src || !this.props.defaultSrc ? this.props.src : this.props.defaultSrc;

    return (
      <div className={classes.join(' ')}>
        {tooltip}
        <div className={previewClasses.join(' ')}>
          <Image src={this.state.previewSrc} />
        </div>
        <div className={currentClasses.join(' ')}>
          <Image src={logoSrc} />
        </div>
        <input
          type="file"
          ref={input => this.logoInput = input}
          onChange={this.onChange.bind(this)}
        />
        <i onClick={this.editLogo.bind(this)} className="edit fa fa-pencil"/>
      </div>
    );
  }

  private previewImage(input: HTMLInputElement) {
    if (input.files && input.files[0]) {
       let reader = new FileReader();
       reader.onload = ((e: any) => {
        this.setState({
          previewSrc: e.target.result
        });
       }).bind(this);

       reader.readAsDataURL(input.files[0]);
    }
  }

  private resetError(delay = 10) {
    setTimeout(() => {
      this.setState({
        state: '',
        error: undefined,
      });
    }, delay * 1000);
  }
}
