import * as React from 'react';
import { Link } from 'react-router';

import { urlForTournament, logoutAndRedirect } from 'common/helpers';
import { Image } from 'ui/components/image';
import { LoginLink } from 'ui/components/login-link';
import { NavigationMobile } from 'ui/components/navigation-mobile';
import { Placeholder } from 'ui/components/placeholder';
import { UserAvatar } from 'ui/components/user-avatar';
import { TournamentsModel } from 'state/tournaments/model';

import {
  UserModel,
  GameModel,
  TournamentModel,
} from 'tourney-sdk-react';

const styles = require('./styles.scss');

export interface NavigationProps {
  location?: any;
  logoutUser?: Function;
  loading: boolean;
  loggedIn: boolean;
  appLoading: boolean;
  user: UserModel;
  activeTournaments: TournamentModel[];
  activeTournamentIds: string[];
}

export interface NavigationState {
  mobileOpen: boolean;
  initialNav: boolean;
}

const INITIAL_NAV_THRESHOLD = 120;

export class Navigation extends React.Component<any, any> {

  public constructor() {
    super();
    this.state = {
      mobileOpen: false,
      initialNav: true,
    };
    this.handleScroll = this.handleScroll.bind(this);
  }

  public componentDidMount() {
    window.addEventListener('scroll', this.handleScroll);
  }
  public componentWillUnmount() {
    window.removeEventListener('scroll', this.handleScroll);
  }

  public handleScroll(e: any) {
    const shouldUseInitial = window.scrollY <= INITIAL_NAV_THRESHOLD;
    if (shouldUseInitial !== this.state.initialNav) {
      this.setState({
        initialNav: shouldUseInitial
      });
    }
  }

  public toggleAction() {
    this.setState({
      mobileOpen: !this.state.mobileOpen
    });
  }

  public closeMobileNav() {
    this.setState({
      mobileOpen: false
    });
  }

  public render() {
    const { loading, activeTournaments, loggedIn, user, location, logoutUser } = this.props;
    const navClasses = ['main-nav'];
    const fullLogo = require('common/assets/ahgl_full_logo.svg') as string;
    const logo = require('common/assets/ahgl_badge_logo.svg') as string;
    const logoLoading = require('common/assets/logo-loading.svg') as string;
    const mobileLogo = require('common/assets/ahgl_text_logo.svg') as string;
    const loginUserIcon = require('common/assets/person.svg') as string;

    let loginSection: JSX.Element;
    let mobileUser: JSX.Element;
    if (!loggedIn) {
      loginSection = (
        <LoginLink>
          <div className="login-container">
              <img src={loginUserIcon}/> Login
          </div>
        </LoginLink>
      );
      mobileUser = (
        <LoginLink>
          <div className="mobile-user">
            <i className="fa fa-user"/>
          </div>
        </LoginLink>
      );
    } else {
      loginSection = this.renderUserNav();
      mobileUser = (
        <Link to="/settings"  onClick={this.closeMobileNav.bind(this)}>
          <div className="mobile-user">
            <i className="fa fa-user"/>
          </div>
        </Link>
      );
    }

    if (this.props.loading) {
      navClasses.push('loading');
    }

    if (this.state.initialNav) {
      navClasses.push('initial');
    }

    const gameMap = loading ? [] : this.createGameMap(activeTournaments);
    const gameLinks = gameMap.map((mapping) => {
      let containerProps = {
        className: location.pathname.indexOf(urlForTournament(mapping.tournament)) !== -1 ? 'active' : '',
        key: mapping.tournament.id,
      };
      return (
        <li {...containerProps}>
          <Link to={urlForTournament(mapping.tournament)}>{mapping.game.name}</Link>
        </li>
      );
    });

    const toggleAction = this.toggleAction.bind(this);
    const mobileNavProps = {
      open: this.state.mobileOpen,
      toggleAction,
      gameMap,
      myUser: user,
      logoutUser
    };
    const mobileIconClasses = ['fa'];
    mobileIconClasses.push(mobileNavProps.open ? 'fa-times' : 'fa-bars');

    return <div className="nav-containers">
      <nav className={navClasses.join(' ')}>
        <div className="nav-content">
          <Link to="/">
            <img src={fullLogo} className="nav-logo desktop" />
            <img src={logoLoading} className="nav-logo loading"/>
          </Link>
          <img src={mobileLogo} className="nav-logo mobile" />
          <ul className="nav-items">
            {gameLinks}
            <li className={location.pathname === '/charities' ? 'active' : ''}>
              <Link to="/charities">Charities</Link>
            </li>
            <li className={location.pathname === '/faq' ? 'active' : ''}>
              <Link to="/faq">FAQ</Link>
            </li>
          </ul>
          <div className="nav-user">
            {loginSection}
          </div>

          <div className="mobile-bars" onClick={toggleAction}>
            <i className={mobileIconClasses.join(' ')}/>
          </div>
          {mobileUser}
        </div>
      </nav>
      <NavigationMobile {...mobileNavProps}/>
    </div>;
  }

  private createGameMap(tournaments: TournamentModel[]): {tournament: TournamentModel, game: GameModel}[] {
    // slice is a limit for testing limitations, ideally this call will only include
    // one of each game, and a reasonable amount to show in nav.
    return tournaments.slice(0, 5).map((tournament) => {
      return {
        tournament,
        game: tournament.game,
      };
    });
  }

  private renderUserNav(): JSX.Element {
    const { user, logoutUser } = this.props;
    const logout = () => {
      logoutAndRedirect(logoutUser, '/');
    };

    return (
      <div>
        <Link to="/settings">
          <div className="avatar">
            <UserAvatar avatar={user.avatar} />
          </div>
          <div className="username">{user.display_name}</div>
        </Link>
        <i className="sign-out fa fa-sign-out" onClick={logout} />
      </div>
    );
  }
}
