import * as React from 'react';
import * as moment from 'moment';
import { keys, every, some, sortBy } from 'lodash';
import { Link } from 'react-router';
import {
  SeriesModel,
  MatchModel,
  OpponentModel,
  UserModel,
} from 'tourney-sdk-react';

import { ICONS } from 'common/constants';
import { Modal } from 'ui/components/modal';
import { Contender } from 'ui/components/tournament/contender';
import { Image } from 'ui/components/image';
import { Button } from 'ui/components/button';
import { Tooltip } from 'ui/components/tooltip';
import { ReportingMatch } from 'ui/components/reporting/match';

const styles = require('./styles.scss');

const SECTIONS = {
  Results: {
    name: 'Results',
    iconName: 'trophy',
  },
  Chat: {
    name: 'Chat',
    iconName: 'chat',
  }
};
type validSections = 'Results' | 'Chat';

interface SeriesReportingProps {
  seriesId: string;
  series?: SeriesModel;
  loading?: boolean;
  closeModal: any;
  getSeries: Function;
  updateMatch: Function;
  user: UserModel;
}

interface SeriesReportingState {
  selectedSection?: validSections;
  winners?: {[key: string]: string};
  updatingMatches?: boolean;
  error?: undefined | string;
}

export class ReportingSeriesComponent extends React.Component<SeriesReportingProps, SeriesReportingState> {
  public constructor(props: any) {
    super(props);
    this.state = {
      selectedSection: 'Results',
      winners: {},
      updatingMatches: false,
      error: undefined,
    };
  }

  public render() {
    const { series, loading, closeModal } = this.props;

    if (loading) {
      return <div/>;
    }

    const sortedOpponents = sortBy(series.opponents, (opp: OpponentModel) => opp.contender_id);
    let contenders: JSX.Element[] = sortedOpponents.map((opp: OpponentModel) => {
      return (
        <Contender
          key={opp.contender.id}
          contender={opp.contender}
          withName={false}
        />
      );
    });
    contenders.splice(1, 0, <div key={0} className="vs">vs</div>);

    let subsections = [SECTIONS.Results, SECTIONS.Chat].map((section: any) => {
      let classes = ['sub-section'];
      if (section.name === this.state.selectedSection) {
        classes.push('active');
      }
      return (
        <section key={section.name} className={classes.join(' ')} onClick={this.selectSection.bind(this, section.name)}>
          <Image src={ICONS[section.iconName]} /> {section.name}
        </section>
      );
    });

    let content = this.state.selectedSection === 'Results'
      ? this.renderReporting()
      : this.renderChatSection();

    return (
      <Modal classNames={['series-modal']}>
        <div className="modal-heading">
          <div className="contenders">
            {contenders}
          </div>
          <div className="contest-info">
            <Image src={ICONS.game} /> <span>{series.game.name}</span>
            <Image src={ICONS.calendar} /> <span>{moment(series.matches[0].scheduled_at).format('MMM D [@] h:mm A')}</span>
          </div>
          <img src={ICONS.x} className="close-modal" onClick={closeModal} />
        </div>
        <div className="modal-content">
          <div className="sub-sections">
            {subsections}
          </div>
          <div className="reporting">
            {content}
          </div>
        </div>
      </Modal>
    );
  }

  private selectSection(sectionName: validSections) {
    this.setState({
      selectedSection: sectionName
    });
  }

  private renderChatSection() {
    return (
      <div className="chat-section">
        <h2>Contact your opponent</h2>
        <div className="url-container">
          <span>Copy and URL and paste it in your Twitch Client to join the server</span>
          <input type="text" value="https://invite.twitch.tv/tR0l13D"/>
        </div>
        <div className="download-client">
          <a href="https://twitch.tv">Download </a> The Twitch Client
        </div>
      </div>
    );
  }

  private renderReporting() {
    const { series, user } = this.props;
    const sortedMatches = sortBy(series.matches, (m: MatchModel) => m.scheduled_at);
    let matches = sortedMatches.map((match: MatchModel, idx: number) => {
      return (
        <ReportingMatch
          key={match.id}
          user={user}
          match={match}
          matchNum={idx + 1}
          winners={this.state.winners}
          selectWinner={this.selectWinner.bind(this)}
        />);
    });

    let buttonProps: {[key: string]: any} = { primary: true };
    if (this.state.updatingMatches) {
      buttonProps.loading = true;
    } else if (this.state.error !== undefined ) {
      buttonProps.error = true;
    }

    let tooltip;
    if (this.state.error !== undefined ) {
      tooltip = <Tooltip position="top">{this.state.error}</Tooltip>;
    }

    const lockedMap = series.matches.map((match: MatchModel) => match.state === 'locked' || match.state === 'decided');
    let submitButton = undefined;
    let headingText = <h2>Results</h2>;
    if (!every(lockedMap)) {
      submitButton = <Button {...buttonProps} onClick={this.updateWinners.bind(this)}>Submit Results</Button>;
      headingText = <h2>Select Winners</h2>;
    } else {
      // TODO: Results built now will not be accurate for SC2
      //        not sure if it is worth building out here.
      // we need to find out X : X with Winner
      // let winsMap = {};
      // let teams = {};
      // series.matches.forEach((match: MatchModel) => {

      // });
    }

    const contestedMap = series.matches.map((match: MatchModel) => match.state === 'contested');
    let underReview = undefined;
    if (some(contestedMap)) {
      headingText = undefined;
      underReview = (
        <div className="under-review">
          <h3>Pending Admin Review</h3>
          <span>Conflicting results were submitted. Admins will be reviewing this match</span>
        </div>
      );
    }

    return (
      <div className="reporting-section">
        {headingText}
        {underReview}
        <div className="matches-container">
          {matches}
        </div>
        <div className="button-container">
          {tooltip}
          {submitButton}
        </div>
      </div>
    );
  }

  private selectWinner(matchId: string, opponentId: string) {
    this.setState({
      ...this.state,
      error: undefined,
      winners: {
        ...this.state.winners,
        [matchId]: opponentId,
      }
    });
  }

  private updateWinners() {
    const { winners } = this.state;
    const updatePromises = keys(winners).map((matchId: string) => {
      return this.props.updateMatch({ id: matchId, winner_id: winners[matchId] });
    });
    this.setState({
      error: undefined,
      updatingMatches: true
    });
    Promise.all(updatePromises)
      .then(() => {
        this.setState({
          winners: {},
          updatingMatches: false,
        });
      })
      .catch((err) => {
        this.setState({
          updatingMatches: false,
          error: err.data.error,
        });
        this.clearError();
      });
  }

  private clearError(delay = 10) {
    setTimeout(() => {
      this.setState({
        error: undefined
      });
    }, delay * 1000);
  }
}
