import * as React from 'react';
import { Link } from 'react-router';
import { TournamentModel, TeamModel } from 'tourney-sdk-react';

import { AppModel } from 'state/model';
import { Button } from 'ui/components/button';
import { Tooltip } from 'ui/components/tooltip';
import { TeamLogo } from 'ui/components/team/logo';

let styles = require('./styles/styles.scss');

export interface TournamentProps {
  register: Function;
  create: Function;
  tournament: TournamentModel;
  teams: TeamModel[];
}

interface TeamSelectState {
  teamName: string;
  creating: Boolean;
  selectedTeam: TeamModel | undefined;
  dropDownOpen: Boolean;
  loading: boolean;
}

export class TeamSelectComponent extends React.Component<TournamentProps, any> {
  private validTeams: TeamModel[];
  public componentWillMount() {
    const teams = this.props.teams || [];
    this.state = {
      teamName: '',
      creating: teams.length === 0,
      selectedTeam: teams[0] || undefined,
      dropDownOpen: false,
      regError: undefined,
      loading: false,
    };
  }

  public render() {
    const { tournament } = this.props;
    // selected can either be selected Team, or input for create team
    const actionText = this.state.creating ? 'CREATE' : 'REGISTER';
    const buttonProps = {
      onClick: this.registerAction.bind(this),
      connect: Button.CONNECTS.left,
      loading: this.state.loading,
    };
    const iconClasses = ['fa', this.state.creating ? 'fa-plus' : 'fa-arrow-right'];
    let tooltip;
    if (this.state.regError) {
      tooltip = <Tooltip position="top">{this.state.regError}</Tooltip>;
    }

    return (
      <div className="team-select">
        {tooltip}
        {this.renderSelectedTeam()}
        <Button {...buttonProps}>
          <i className={iconClasses.join(' ')}/><span className="action-text">{actionText}</span>
        </Button>
        {this.renderDropDown()}
      </div>
    );
  }

  private findTeamByName(name: string) {
    return this.props.teams.find((team) => team.name === name);
  }

  private renderSelectedTeam(): JSX.Element {
    let currentlySelected: JSX.Element;
    if (this.state.creating) {
      currentlySelected = (
        <input
          type="text"
          name="teamName"
          className="create-team"
          placeholder="New Team Name"
          value={this.state.teamName}
          onChange={this.updateForm.bind(this)}
        />
      );
    } else {
      currentlySelected = (
        <div className="selected-team" onClick={this.toggleDropDown.bind(this)}>
          <TeamLogo team={this.state.selectedTeam} />
          {this.state.selectedTeam.name}
        </div>
      );
    }

    let arrowElement: JSX.Element = undefined;
    if (this.props.teams && this.props.teams.length > 0) {
      let arrowDirection = this.state.dropDownOpen ? 'fa-long-arrow-up' : 'fa-long-arrow-down';
      arrowElement = (
        <div className="dropdown-arrow" onClick={this.toggleDropDown.bind(this)}>
          <i className={['fa', arrowDirection].join(' ')}/>
        </div>);
    }

    let classes = ['currently-selected'];
    if (this.state.dropDownOpen) {
      classes.push('open');
    }

    return (
      <div className={classes.join(' ')}>
        {currentlySelected}
        {arrowElement}
      </div>
    );
  }

  private registerAction(event: any) {
    if (this.state.creating) {
      return this.createTeamAndRegister();
    } else {
      return this.registerTeam();
    }
  }

  private registerTeam() {
    const data = {
      tournament_id: this.props.tournament.id,
      team_id: this.state.selectedTeam.id
    };
    this.setState({
      loading: true,
    });
    return this.props.register(data)
      .catch((err: any) => {
        this.setState({
          regError: err.data.error,
          loading: false,
        });
      });
  }

  private createTeamAndRegister() {
    if (this.state.teamName === '') {
      return false;
    }
    this.setState({
      loading: true
    });
    return this.props
      .create({
        name: this.state.teamName
      })
      .then((result: TeamModel) => {
        let team = this.findTeamByName(result.name);
        this.setState({
          creating: false,
          teamName: '',
          selectedTeam: team,
          loading: false,
        });
        return team;
      })
      .then((team: TeamModel) => {
        return this.registerTeam();
      })
      .catch((err: any) => {
        this.setState({
          regError: err.data.error,
          loading: false,
        });
      });
  }

  private updateForm(event: Event) {
    const target = event.target as HTMLInputElement;
    this.setState({
      [target.name]: target.value
    });
  }

  private renderDropDown(): JSX.Element {
    const teams = this.props.teams || [];

    if (teams.length === 0 || !this.state.dropDownOpen) {
      return undefined;
    }

    let allTeams: JSX.Element[] = teams.map((t) => {
      let selectTeam = this.selectTeam.bind(this, t);
      return (
        <li key={t.id} className="selectable-team" onClick={selectTeam}>
          <TeamLogo team={t} />
          {t.name}
        </li>
      );
    });

    return (
      <ul className="drop-down">
        {allTeams}
        <li className="create-new" onClick={this.createNewTeam.bind(this)}>
          <i className="fa fa-pencil"/>Create New Team
        </li>
      </ul>
    );
  }

  private createNewTeam(): void {
    this.setState({
      selectedTeam: undefined,
      dropDownOpen: false,
      creating: true,
      teamName: '',
      regError: undefined,
    });
  }

  private selectTeam(team: TeamModel): void {
    this.setState({
      selectedTeam: team,
      dropDownOpen: false,
      creating: false,
      regError: undefined,
    });
  }

  private toggleDropDown(): Boolean {
    this.setState({
      dropDownOpen: !this.state.dropDownOpen,
      regError: undefined,
    });
    return this.state.dropDownOpen;
  }
}
