import * as React from 'react';
import { Link } from 'react-router';
import { TeamModel, UserModel, TournamentEntryModel } from 'tourney-sdk-react';

import { ICONS } from 'common/constants';
import { urlForTournament } from 'common/helpers';
import { Button } from 'ui/components/button';
import { TeamInvite } from 'ui/components/team/invite';
import { UserAvatar } from 'ui/components/user-avatar';
import { Tooltip } from 'ui/components/tooltip';
import { TooltipHover } from 'ui/components/tooltip-hover';

import { Placeholder, Image, ImageUpload } from 'ui/components';
import {
  Tournament,
} from 'common/constants';
import { AppModel } from 'state/model';

interface TeamManageProps {
  team: TeamModel;
  updateTeam: Function;
  deleteTeamUser: Function;
  deleteTeamInvitation: Function;
  getTeam: Function;
  openGameModal: Function;
  myUser: UserModel;
}

interface TeamManageState {
  showingLink?: boolean;
  leaveTournamentError?: string;
  leaveLoading?: boolean;
}

const styles = require('./styles.scss');

export class TeamManageComponent extends React.Component<TeamManageProps, TeamManageState> {

  public componentWillMount() {
    this.state = {
      showingLink: false,
      leaveTournamentError: undefined,
      leaveLoading: false
    };
  }

  /* tslint:disable: cyclomatic-complexity */
  public render() {
    const defaultTeamIcon = require('common/assets/def_team.png') as string;
    const { team, myUser } = this.props;
    // use team_members to find important roles
    let captainId = '';
    if (team.team_members) {
      const ownerMember = team.team_members.filter((member: any) => {
        return member.roles.length > 0 && member.roles.indexOf('owner') !== -1;
      });
      if (ownerMember.length > 0) {
        captainId = ownerMember[0].user_id;
      }
    }
    const players: JSX.Element[] = team.users.map(
      (user: UserModel, idx: number) => this.renderSinglePlayer(user, idx, captainId));
    const invitations: JSX.Element[] = team.team_invitations.map(
      (invitation: any, idx: number) => this.renderInvitation(invitation, idx, captainId));
    const toggleShare = this.toggleShareLink.bind(this);
    const selectText = (event: any) => {
      event.target.setSelectionRange(0, event.target.value.length);
    };
    let shareState: JSX.Element;
    if ( myUser.id === captainId ){
      if (this.state.showingLink) {
        shareState = <div className="share-link">
          <TeamInvite
            teamId={team.id}
          />
        </div>;
      } else {
        shareState = <div className="add-player" onClick={toggleShare}>
          <img className="empty-add-icon" src={ICONS.addbuddy} />
        </div>;
      }
    }

    let entryInformation: JSX.Element;
    let actionButton: JSX.Element;
    let teamManageClasses = ['team-manage'];
    const numUsers = 6;
    const verifiedClass = team.users.length === numUsers
      ? 'verified'
      : '';
    const verifiedTooltip = <Tooltip position='top' type='info'>You must have 6 verified players on your team to compete.</Tooltip>;
    let verifiedMembers = (
      <li className={verifiedClass}>
        <TooltipHover tooltip={verifiedTooltip}>
          <div className="cond-label">
            <img src={ICONS.group} /> Verfied
          </div>
          <div className="cond-amount">
            {team.users.length}/{numUsers}
          </div>
        </TooltipHover>
      </li>
    );
    let competingFee: JSX.Element;
    let gameBox: JSX.Element;

    if (team.tournament_entries.length > 0) {
      const firstEntry = team.tournament_entries[0];
      const paymentsReceived = team.tournament_entries.
        filter((e: any) => e && e.flags.indexOf('payment_received') !== -1);
      const paymentReceived = paymentsReceived.length === team.tournament_entries.length;
      if(firstEntry.tournament.game){
        teamManageClasses.push(firstEntry.tournament.game.abbr);
        gameBox = <Image src={firstEntry.tournament.game.cover.preview} className="game-box"/>;
      }
      teamManageClasses.push('has-entry');
      let feeClass: string;
      let feeText: string;
      const feeTooltip = <Tooltip position="top" type="info">Registration Fee is required to compete once all members have been verified.</Tooltip>;

      if (firstEntry.flags.indexOf('payment_received') !== -1) {
        feeClass = 'verified';
        feeText = 'PAID';
      } else {
        feeText = '$300';
      }
      competingFee = (
        <li className={feeClass}>
          <TooltipHover tooltip={feeTooltip}>
            <Link to={urlForTournament(firstEntry.tournament)}>
              <div className="cond-label">
                <img src={ICONS.creditcard} /> Registration fee
              </div>
              <div className="cond-amount">
                {feeText}
              </div>
            </Link>
          </TooltipHover>
        </li>
      );
      let buttonProps = {
        primary: true,
        onClick: this.leaveTournament.bind(this, firstEntry),
        error: !!this.state.leaveTournamentError,
        loading: this.state.leaveLoading,
      };

      actionButton = (
        <Button {...buttonProps}>
          Leave
        </Button>
      );
    } else {
      actionButton = (
        <Button primary onClick={this.props.openGameModal}>
          Compete
        </Button>
      );
    }

    const tooltip = this.state.leaveTournamentError
      ? <Tooltip position="top">{this.state.leaveTournamentError}</Tooltip>
      : undefined;

    entryInformation = (
      <div className="manage-entry">
        <ul className="conditions">
          {verifiedMembers}
          {competingFee}
          <li className="button-container">
            {tooltip}
            {actionButton}
          </li>
        </ul>
        {gameBox}
      </div>
    );

    return (
      <div className={teamManageClasses.join(' ')}>
        <div className="manage-heading">
          <ImageUpload
            className="team-logo"
            defaultSrc={defaultTeamIcon}
            src={team.logo.thumb}
            onChange={this.uploadLogo.bind(this)}
          />
          <div className="team-name">
            {team.name}
          </div>
        </div>
        {entryInformation}
        <div className="player-list">
          {players}
          {invitations}
          {shareState}
        </div>
      </div>
    );
  }
  /* tslint:enable: cyclomatic-complexity */

  public uploadLogo(e: Event) {
    let team = this.props.team;
    return this.props.updateTeam(
      {
        id: team.id,
        logo: e.target.files[0],
      });
  }

  public deleteTeamUser(id: Number) {
    return this.props.deleteTeamUser(this.props.team.id, id)
      .then((user: UserModel) => {
        return this.props.getTeam(this.props.team.id);
      });
  }

  public deleteTeamInvitation(id: Number) {
    return this.props.deleteTeamInvitation(id);
  }

  public leaveTournament(entry: TournamentEntryModel) {
    if (confirm('Are you sure you want to leave this tournament?')) {
      this.setState({
        leaveLoading: true
      });
      this.props.deleteTournamentEntry(entry)
        .catch((error: any) => {
          this.setState({
            leaveTournamentError: error.data.error,
            leaveLoading: false
          });
          this.resetError();
        });
    }
  }

  public renderInvitation(invitation: any, idx: any, captainId: string): JSX.Element {
    const { myUser } = this.props;
    let classes = ['single-player', 'invitation'];

    let ownerControls: JSX.Element;
    if (myUser.id === captainId) {
      ownerControls = (
        <div className="icons">
          <img src={ICONS.removeplayerx} className="icon icon-remove" onClick={this.deleteTeamInvitation.bind(this, invitation.id)}/>
        </div>
      );
    }

    if (invitation) {
      return <div className={classes.join(' ')} key={invitation.id.toString()}>
        <div className="avatar"><UserAvatar avatar={undefined}/></div>
        <div className="username" alt={invitation.email + ' (pending)'}>{invitation.email} (pending)</div>
        {ownerControls}
      </div>;
    }

    classes.push('loading');
    return <div className={classes.join(' ')} key={idx} />;
  }

  public renderSinglePlayer(user: UserModel, idx: any, captainId: string): JSX.Element {
    const { myUser } = this.props;
    let classes = ['single-player', 'player'];

    if (user) {
      let captainIcon: JSX.Element = undefined;
      let playerIcons: JSX.Element[] = [];
      if (user.id === captainId) {
        captainIcon = <img src={ICONS.star} className="icon icon-captain"/>;
      }

      if (myUser.id === captainId && myUser.id !== user.id) {
        playerIcons.push(
          <img src={ICONS.removeplayerx} className="icon icon-remove" onClick={this.deleteTeamUser.bind(this, user.id)}/>
        );
      }

      return <div className={classes.join(' ')} key={[this.props.team.id, user.id].join('-')}>
        <div className="avatar">
          <UserAvatar avatar={user.avatar} />
        </div>
        
        <div className="username">
          {captainIcon}
          {user.display_name}
        </div>
        <div className="icons">
          {/* verified support isn't ready yet */}
          {/*<img src={ICONS.verified_check} className="icon icon-verified"/>*/}
          {playerIcons}
        </div>
      </div>;
    }

    classes.push('loading');
    return <div className={classes.join(' ')} key={idx} />;
  }

  private toggleShareLink(): Boolean {
    const shareState = !this.state.showingLink;
    this.setState({
      showingLink: shareState
    });
    return shareState;
  }

  private resetError(delay = 10) {
    setTimeout(() => {
      this.setState({
        leaveTournamentError: undefined
      });
    }, delay * 1000);
  }
}
