import * as React from 'react';
import * as _ from 'lodash';
import { config } from 'config';
import { Link } from 'react-router';
import { TournamentModel, TournamentEntryModel, UserModel } from 'tourney-sdk-react';

import { loadScript } from 'common/helpers';
import { Button } from 'ui/components/button';
import { Placeholder } from 'ui/components/placeholder';
import { TeamLogo } from 'ui/components/team/logo';
import { Tooltip } from 'ui/components/tooltip';
import { TooltipHover } from 'ui/components/tooltip-hover';

const styles = require('./styles.scss');

export interface EntryRequirements {
  playerCount: number;
  payment?: boolean;
}

export interface TournamentEntryRequirementsProps {
  entry: TournamentEntryModel;
  requirements: EntryRequirements;
  deleteTournamentEntry: Function;
}

interface TournamentEntryRequirementsState {
  buttonStatus?: undefined | 'error' | 'loading';
  error?: string;
  paypalButtonStatus?: undefined | 'scriptLoading' | 'scriptLoaded' | 'buttonPlaced';
}

export class TournamentEntryRequirementsComponent extends React.Component<any, TournamentEntryRequirementsState> {
  public constructor(props: any) {
    super(props);
    this.state = {
      buttonStatus: undefined,
      error: undefined,
      paypalButtonStatus: undefined,
    };
  }

  public componentDidUpdate(lastProps: any, lastState: any) {
    // check for paypal button
    const { requirements, team, user, loading } = this.props;
    const { paypalButtonStatus } = this.state;
    const paymentOkay = this.props.entry.flags.indexOf('payment_received') !== -1;
    let hasVerifiedTeam = false;
    let isOwner = false;
    if (team.team_members) {
      hasVerifiedTeam = team.users.length === requirements.playerCount;
      const ownerMember = team.team_members.filter((member: any) => {
        return member.roles.length > 0 && member.roles.indexOf('owner') !== -1;
      });
      if (ownerMember.length > 0) {
        isOwner = ownerMember[0].user_id === user.id;
      }
    }
    if (hasVerifiedTeam && !paymentOkay && isOwner) {
      if (paypalButtonStatus === undefined) {
        this.addPaypalScript();
        return;
      }

      if (!loading && paypalButtonStatus === 'scriptLoaded') {
        this.addPaypalButton();
      }
    }
  }

  public get loading() {
    return this.props.loading || !_.every(this.props.team.users, (user) => user !== undefined);
  }

  /* tslint:disable: cyclomatic-complexity */
  public render() {
    const { team, entry, requirements, loading } = this.props;
    let content: JSX.Element = <div />;
    let containerClasses = ['tournament-requirements'];

    if (!this.props.loading) {
      // const verifiedPlayers = team.users.filter((user: UserModel) => {
      //   return _.isEmpty(user.organization_ids) || user.email_new !== '';
      // });
      const verifiedPlayers: UserModel[] = team.users;
      const playerCountOkay = verifiedPlayers.length === requirements.playerCount;
      const paymentOkay = entry.flags.indexOf('payment_received') !== -1;
      const everythingOkay = playerCountOkay && paymentOkay;

      let paymentClasses = ['req-content'];
      let paymentContent = <span>$300 <div id="paypal-button"/></span>;
      if (paymentOkay) {
        paymentClasses.push('success');
        paymentContent = <span><i className="fa fa-check" />Paid</span>;
      } else {
        paymentClasses.push('error');
      }

      let verifiedClasses = ['req-content'];
      if (playerCountOkay) {
        verifiedClasses.push('success');
      } else {
        verifiedClasses.push('error');
      }

      const messageText = everythingOkay
        ? 'Everything is in order.'
        : 'Please verify your players and registration fee.';

      let buttonProps = {
        [this.state.buttonStatus]: true,
        secondary: true,
        size: Button.SIZES.small,
        onClick: this.leaveTournament.bind(this, entry)
      };

      // leave button tooltip
      let tooltip;
      if (this.state.error) {
        tooltip = <Tooltip position="top">{this.state.error}</Tooltip>;
      }
      let verifiedHover = <Tooltip position="top" type="info">You must have {requirements.playerCount} verified players on your team to compete.</Tooltip>;
      let paymentHover = <Tooltip position="top" type="info">Registration Fee is required to compete once all members have been verified.</Tooltip>;

      content = <div>
        <div className="left-content">
          <div className="logo-container">
            <TeamLogo team={team} />
            <div className={['status-icon', everythingOkay ? 'success' : 'error'].join(' ')}/>
          </div>
          <div className="team-text">
            <div className="team-title">{team.name}</div>
            <div className={['message', everythingOkay ? 'success' : 'error'].join(' ')}>
              {messageText}
            </div>
          </div>
        </div>
        <div className="right-content players">
          <TooltipHover tooltip={verifiedHover}>
            <div className="label">
              Players Verified
            </div>
            <div className={verifiedClasses.join(' ')}>
              <span>{verifiedPlayers.length}</span>/{requirements.playerCount}
            </div>
          </TooltipHover>
        </div>
        <div className="right-content payment">
          <TooltipHover tooltip={paymentHover}>
            <div className="label">
              Registration Fee
            </div>
            <div className={paymentClasses.join(' ')}>
              {paymentContent}
            </div>
          </TooltipHover>
        </div>
        <div className="leave-container">
          {tooltip}
          <Button {...buttonProps}>Leave</Button>
        </div>
      </div>;
    } else {
      containerClasses.push('loading');
    }

    return (
      <section className={containerClasses.join(' ')}>
        {content}
      </section>
    );
  }
  /* tslint:enable: cyclomatic-complexity */

  private addPaypalScript() {
    this.setState({ paypalButtonStatus: 'scriptLoading' });
    if (!document.getElementById('paypal-script')) {
      loadScript('paypal-script', 'https://www.paypalobjects.com/api/checkout.js')
        .then(() => {
          this.setState({ paypalButtonStatus: 'scriptLoaded' });
        });
    } else {
      this.setState({ paypalButtonStatus: 'scriptLoaded' });
    }
  }

  private leaveTournament(entry: TournamentEntryModel) {
    if (confirm('Are you sure you want to leave this tournament?')) {
      this.setState({
        buttonStatus: 'loading'
      });
      this.props.deleteTournamentEntry(entry)
        .catch((error: any) => {
          this.setState({
            buttonStatus: 'error',
            error: error.data.error
          });
          this.resetError();
        });
    }
  }

  private addPaypalButton() {
    const { entry, updateTournamentEntry, user } = this.props;

    paypal.Button.render({
      env: config.paypalEnv,
      client: {
        sandbox: config.paypalSandboxClientId,
        production: config.paypalProductionClientId
      },
      style: {
        label: 'checkout',
        size:  'tiny',
        shape: 'rect',
        color: 'silver'
      },
      payment() {
        let env = this.props.env;
        let client = this.props.client;
        const customInfo = {
          entry_id: entry.id,
          team_id: entry.entrant.id,
          tournament_id: entry.tournament.id,
          user_id: user.id,
        };

        return paypal.rest.payment.create(env, client, {
          transactions: [{
            amount: {
              total: '300.00',
              currency: 'USD',
            },
            description: 'Team Entry',
            custom: JSON.stringify(customInfo),
            item_list: {
              items: [{
                name: `AHGL ${entry.entrant.name} Entry Fee`,
                description: `Team Entry: ${entry.entrant.name}`,
                price: '300.00',
                currency: 'USD',
                quantity: 1
              }]
            },
          }],
        });
      },
      commit: true,
      onAuthorize(data: any, actions: any) {
          return actions.payment.execute().then(() => {
            // Show a success page to the buyer
            // this should be verfified with the api
            updateTournamentEntry({
              id: entry.id,
              flags: ['payment_received'],
            });
          });
      }
    }, '#paypal-button');

    this.setState({ paypalButtonStatus: 'buttonPlaced' });
  }

  private resetError(delay = 10) {
    setTimeout(() => {
      let nextState: TournamentEntryRequirementsState = {
        error: undefined,
      };
      if (this.state.buttonStatus === 'error') {
        nextState.buttonStatus = undefined;
      }
      this.setState(nextState);
    }, delay * 1000);
  }
}
