import * as _ from 'lodash';
import * as React from 'react';
import { TournamentModel } from 'tourney-sdk-react';

import { AppModel } from 'state/model';
import { HomeIntro } from 'ui/components/home/intro';
import { HeroSlider } from 'ui/components/hero-slider';
import { Hero } from 'ui/components/hero';
import { TournamentCard } from 'ui/components/tournament/card';

import { HomeStateProps } from './index';

const styles = require('./styles.scss');

interface Slide {
  type: 'tournament' | 'intro';
  data?: TournamentModel;
  content: React.ReactNode;
}

// Used to fill space to avoid space jump on loading
// This being hardcoded is not ideal, but this should probably be solved by seeding redux
const MIN_TOURNAMENTS = 3;
const ANIMATION_SPEED = 0.5;
const ANIMATION_DELAY = 15;

export class HomePageComponent extends React.Component<HomeStateProps, any> {
  private interval: number;

  public constructor(props: HomeStateProps) {
    super(props);
    this.state = {
      slide: 0,
      timer: false,
    };
  }

  public componentDidUpdate(lastProps: HomeStateProps) {
    if (this.props.activeTournaments !== lastProps.activeTournaments
        && lastProps.activeTournaments === undefined) {
      this.setState({
        slide: 0,
        timer: true
      });
      this.resetInterval();
    }
  }

  public render() {
    let { activeTournaments, activeTournamentIds, loading } = this.props;
    const slides = this.allSlides();

    const slide = slides[this.state.slide];
    activeTournaments = loading ? _.range(MIN_TOURNAMENTS).map(() => undefined) : activeTournaments;
    const tournamentCards: JSX.Element[] = activeTournaments.map((tournament: TournamentModel, idx: number): JSX.Element => {
      const props = {
        key: idx,
        tournament,
        loading: tournament === undefined,
        active: slide.type === 'tournament' && slide.data === tournament,
        headerClick: this.setSlide.bind(this, idx + 1),
      };
      return (
        <TournamentCard {...props} />
      );
    });

    return (
      <div>
        <HeroSlider content={slide.content} animationDuration={ANIMATION_SPEED}/>
        <div className="page-container home-page">
          <section className="tournament-cards">
            {tournamentCards}
          </section>
        </div>
      </div>
    );
  }

  private resetInterval() {
    clearInterval(this.interval);
    this.interval = setInterval(() => {
      this.nextSlide();
    }, ANIMATION_DELAY * 1000);
  }

  private setSlide(slideNum: number) {
    this.setState({
      slide: slideNum
    });
    this.resetInterval();
  }

  private nextSlide() {
    let nextSlide = this.state.slide + 1;
    if (nextSlide >= this.allSlides().length) {
      nextSlide = 0;
    }
    this.setState({
      slide: nextSlide,
    });
  }

  private allSlides(): Slide[] {
    const { loading, activeTournaments } = this.props;
    const introSlide: Slide = {
      type: 'intro',
      content: <HomeIntro key={0} />
    };
    const tournaments = loading ? [] : activeTournaments;
    const tournamentHeroes = tournaments.map((tournament, idx): Slide => {
      return {
        type: 'tournament',
        data: tournament,
        content: <Hero key={idx + 1} idx={idx} tournament={tournament} />,
      };
    });
    return [introSlide, ...tournamentHeroes];
  }

}
