import * as React from 'react';
import { sortBy } from 'lodash';
import { TeamModel } from 'tourney-sdk-react';

import { urlForSettingsSection } from 'common/helpers';
import { SETTING_SECTIONS } from 'common/constants';
import { AppModel } from 'state/model';
import { EmailVerification } from 'ui/components/email-verification';
import { Image } from 'ui/components/image';
import { SubNav } from 'ui/components/sub-nav';
import { UserAvatar } from 'ui/components/user-avatar';

const styles = require('./styles.scss');

export interface SettingsPageProps {
  loggedIn: Boolean;
  user: any;
  location: Location;
  router: any;
}

export class SettingsPageComponent extends React.Component<any, any> {
  public constructor(...args: any[]) {
    super(...args);
    this.state = {
      emailVerification: { loading: false },
      teamInvitation: { loading: false },
    };
  }

  // For lack of a better name or location, this is called on mount and prop
  // changes to route.
  public reroute(props: any) {
    if (props.route.path.match(/^\/settings\/verify/)) {
      const verify = props.verifyEmailToken(props.params.email_token);
      if (verify) {
        this.setState({ emailVerification: { loading: true } });
        verify.then(() => {
          // this is useful to keep from breaking the user refreshing but this isn't
          // ideal still, after initial load, user's new email doesn't show
          props.router.replace('/settings');
          this.setState({ emailVerification: { loading: false } });
        });
      }
    }

    if (props.route.path.match(/^\/settings\/accept/)) {
      if (props.loggedIn) {
        this.acceptInvitation(props.params.token);
      } else {
        localStorage.setItem('team_invitation_token', props.params.token);
        props.authUser();
      }
    }
  }

  public componentWillReceiveProps(nextProps: any) {
    if (this.props.route.path !== nextProps.route.path) {
      this.reroute(nextProps);
    }
  }

  public componentDidMount() {
    this.reroute(this.props);
  }

  // consumes pending team invitations
  public componentDidUpdate(prevProps: any, prevState: any) {
    const acceptToken = localStorage.getItem('team_invitation_token');
    if (acceptToken && this.props.loggedIn && this.props.user) {
      localStorage.removeItem('team_invitation_token');
      this.acceptInvitation(acceptToken);
    }
  }

  public render() {
    if (!this.props.loggedIn) {
      return <div>NOT LOGGED IN</div>;
    }

    return (
      <div className="user-container">
        {this.renderUserHeading()}
        <SubNav
          location={location}
          urlHelper={urlForSettingsSection}
          sections={SETTING_SECTIONS}
        />
        <div className="page-container">
          <EmailVerification
            loading={this.state.emailVerification.loading}
            user={this.props.user}
          />
        </div>
        {this.props.children}
      </div>
    );
  }

  private acceptInvitation(token: string) {
    const accept = this.props.acceptTeamInvitation(token);
    if (accept) {
      this.setState({ teamInvitation: { loading: true } });
      accept.then(() => {
        this.props.router.replace('/settings');
        this.setState({ teamInvitation: { loading: false } });
      })
      .catch((err: any) => {
        throw new Error(err);
      });
    }
  }

  private renderUserHeading(): JSX.Element {
    const { user, location } = this.props;
    const backgroundImage: string = require('common/assets/hexagonpattern.png') as string;
    const bgStyle = { backgroundImage: `url(${backgroundImage})` };
    const mobileStyle = { backgroundImage: `url(${user.avatar})` };
    return (
      <div className="">
        <div className="user-heading" style={bgStyle}>
          <div className="mobile-bg" style={mobileStyle} />
          <div className="page-container heading-content">
            <div className="avatar-container">
              <UserAvatar avatar={user.avatar} />
              <a href="https://twitch.tv/settings/profile">
                <i className="avatar-edit fa fa-pencil"/>
              </a>
            </div>
            <div className="username">
              {user.display_name}
            </div>
          </div>
        </div>
      </div>
    );
  }
}
