import * as React from 'react';
import { sortBy, uniq } from 'lodash';
import { TeamModel, MatchModel, SeriesModel, OpponentModel } from 'tourney-sdk-react';
import * as moment from 'moment';

import { AppModel } from 'state/model';
import { EmailVerification } from 'ui/components/email-verification';
import { Match } from 'ui/components/tournament/match';
import { ReportingSeries } from 'ui/components/reporting/series';

const styles = require('./styles.scss');

export interface SettingsMatchesProps {
  location: any;
}

export class SettingsMatchesComponent extends React.Component<any, any> {
  public constructor(props: any) {
    super(props);

    // ideally we don't have loading state run, but page-wise
    // we cannot be sure both matches and series have been loaded
    // as empty result could be a scenario.
    this.state = {
      seriesDetail: undefined,
      loading: true,
    };
  }

  public componentDidMount() {

    // here is our big content load series -> matches -> teams -> loaded
    if (this.props.loggedIn) {
      // this is not right yet, but series normalizes matches
      // but then matches has no API point to request opponents
      // so for now we need to make sure it also fetches corresponding matches
      this.props.getUserSeries(this.props.user.id)
        .then((seriesIds: string[]) => {
          return this.props.getMatches({ user_id: this.props.user.id });
          // Another approach could be calling get Matches by series, but since this whole page
          // is focused on the user, this is just a single call vs N calls.
          // return Promise.all(
          //   seriesIds.map((seriesId: string) => props.getMatches({ series_id: seriesId }))
          // );
        })
        .then((matches: MatchModel[]) => {
          const teamIds = matches.reduce((memo: string[], match: MatchModel) => {
            return [...memo, ...match.opponents.map((opp: OpponentModel) => opp.contender_id)];
          }, []);
          return this.props.getTeams({ id: uniq(teamIds) });
        })
        .then(() => {
          this.setState({
            loading: false
          });
        });
    }
  }

  // TODO: once it more focused on series > matches, we can start making better states
  // ('ongoing', 'upcoming', 'past', ...)
  public render() {
    const { series, loggedIn } = this.props;
    const { seriesId } = this.state;

    if (!loggedIn) {
      return <div>Must be logged in.</div>;
    }

    if (this.state.loading) {
      return <div className="page-loading"/>;
    }

    const allSeries = series.map((s: SeriesModel) => {
      const match = s.matches[0];
      // this should change once we get new series focused designs.
      // for now we will use matches and open reporting by calling its
      // parent series
      return <Match match={match} key={match.id} openReporting={this.setSeriesDetail.bind(this, s.id)}/>;
    });

    let seriesModal;
    if (seriesId !== undefined) {
      seriesModal = (
        <ReportingSeries
          seriesId={seriesId}
          closeModal={this.closeModal.bind(this)}
        />
      );
    }

    return (
      <div className="page-container matches-page">
        <h2 className="series-section-heading">Ongoing</h2>
        <div className="matches">
          {allSeries}
        </div>
        {seriesModal}
      </div>
    );
  }

  private setSeriesDetail(seriesId: string) {
    this.setState({
      seriesId
    });
  }

  private closeModal() {
    this.setState({
      seriesId: undefined
    });
  }
}
