import * as React from 'react';
import { Link } from 'react-router';
import { TournamentModel, TournamentEntryModel } from 'tourney-sdk-react';
import * as moment from 'moment';

import { urlToTournamentSection } from 'common/helpers';
import { TOURNAMENT_SUB_SECTIONS, ICONS } from 'common/constants';
import { AppModel } from 'state/model';
import { Placeholder } from 'ui/components/placeholder';
import { SignupButton } from 'ui/components/signup-button';
import { TeamSelect } from 'ui/components/team-select';
import { TournamentEntry } from 'ui/components/tournament-entry';
import { TournamentEntryRequirements } from 'ui/components/tournament/entry-requirements';
import { TournamentProgressBar } from 'ui/components/tournament/progress-bar';
import { SubNav } from 'ui/components/sub-nav';


const styles = require('./styles.scss');

interface TournamentDetailState {
  loading: boolean;
}

export class TournamentDetail extends React.Component<any, TournamentDetailState> {
  public constructor(props: any) {
    super(props);
    if (props.tournament === undefined) {
      props.getTournament(props.tournamentId);
    }
  }

  public render() {
    if (this.props.loading) {
      return (<div>Loading</div>);
    }

    const tournament = this.props.tournament;
    let hasEntry;
    let createOrEntry = undefined;
    if (this.props.loggedIn) {
      hasEntry = tournament.tournament_entries.find((entry: any) => {
        return this.props.user.team_ids.indexOf(entry.entrant_id) !== -1;
      });
      createOrEntry = hasEntry ? this.renderTeamEntry(hasEntry) : this.renderTeamRegister(tournament);
    }

    const children = React.Children.map(this.props.children, (child: any) => {
      return React.cloneElement(child, {
        tournament: this.props.tournament
      });
    });

    return (
      <div className="tournament-page">
        {this.renderHeading(tournament)}
        {this.renderSubNav()}
        <div className="page-container">
          {createOrEntry}
          {children}
        </div>
      </div>
    );
  }

  private renderTeamRegister(tournament: TournamentModel): JSX.Element {
    if (tournament.state !== 'registration_open') {
      return <div />;
    }

    const userTeams = this.props.user.teams;

    return (
      <div className="register-team">
        <TeamSelect
          create={this.props.createTeam}
          teams={userTeams}
          tournament={tournament}
          register={this.props.createTournamentEntry}
        />
      </div>
    );
  }

  private renderTeamEntry(entry: TournamentEntryModel): JSX.Element {
    let requirements = {
      playerCount: 2
    };
    let props = {
      key: entry.id,
      entry,
      requirements,
      deleteTournamentEntry: this.props.deleteTournamentEntry,
    };
    return (
      <TournamentEntryRequirements {...props} />
    );
  }

  private renderHeading(tournament: TournamentModel): React.ReactNode {
    const classes = ['tournament-detail-container', `game-${tournament.game.abbr}`];

    return (
      <section className={classes.join(' ')}>
        <div className="page-container">
          <div className="heading-content">
            <div className="game-title">
              {tournament.name}
            </div>
            <Placeholder className="tournament-title">
              Season 1
            </Placeholder>
            <div className="tournament-dates">
              <img src={ICONS.clock} />
              {moment(tournament.starts_at).format('MMM D')} to {moment(tournament.ends_at).format('MMM D')}
            </div>
            <TournamentProgressBar tournament={this.props.tournament}/>
          </div>
          <div className="tournament-game-cover" style={{ backgroundImage: `url(${tournament.game.cover.original})` }}/>
        </div>
      </section>
    );
  }

  private renderSubNav(): JSX.Element {
    const { location, tournament } = this.props;
    return (
      <SubNav
        location={location}
        urlHelper={urlToTournamentSection}
        sections={TOURNAMENT_SUB_SECTIONS}
        model={tournament}
      />
     );
  }
}
