provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-dev/"
}

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "tourney-fig" {
  name = "tourney-fig"
}

data "template_file" "fig_task_definition" {
  template = "${file("task-definitions/tourney-fig.json")}"
  vars {
    name = "tourney-fig"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.tourney-fig.repository_url, "https://", "")}:latest"
    container_port = 80
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "fig_task" {
  family = "tourney-fig"
  container_definitions = "${data.template_file.fig_task_definition.rendered}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=00bcb20fa153152c811b6948d83a7085eaaf50b9"
  name = "tourney-fig"
  prefix = "tourney_fig"
  container_port = "80"
  alb = true
  env = "staging"
  cluster = "tourney"
  certificate_arn = "${module.common.service_certificate_external}"
  task_definition = "${aws_ecs_task_definition.fig_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
}

output "ecr_url" {
  value = "${aws_ecr_repository.tourney-fig.repository_url}"
}
