import * as React from 'react';
import { shallow, mount, render } from 'enzyme';
import { TeamModel } from 'tourney-sdk-react';

import { TeamSelectComponent, TournamentProps } from 'ui/components/team-select/component';

import { TEST_TOURNAMENT, TEST_TEAM } from '../constants';

const registerTeam = jest.fn(() => Promise.resolve({}));
const createTeam = jest.fn(() => Promise.resolve({}));

let props: TournamentProps = {
  register: registerTeam,
  create: createTeam,
  tournament: TEST_TOURNAMENT,
  teams: []
};

let propsWithTeam: TournamentProps = {
  ...props,
  teams: [TEST_TEAM]
};
const newTeamName = 'test';

const ACTION_STATES = {
  CREATE: 'CREATE',
  REGISTER: 'REGISTER',
};

describe('TeamSelect', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('component render', () => {
    it('should render without any teams/register options', () => {
      const component = shallow(
        <TeamSelectComponent {...props} />
      );
      expect(component.find('.action-text').text()).toEqual(ACTION_STATES.CREATE);
      expect(component.find('.drop-down').exists()).toBe(false);
      expect(component.find('.dropdown-arrow').exists()).toBe(false);
    });

    it('should render correctly when given list of teams', () => {
      const component = shallow(
        <TeamSelectComponent {...propsWithTeam} />
      );

      expect(component.find('.action-text').text()).toEqual(ACTION_STATES.REGISTER);
      expect(component.find('.dropdown-arrow').exists()).toBe(true);
      expect(component.find('.drop-down').exists()).toBe(false);
      expect(component.find('.selected-team').text()).toEqual(`<TeamLogo />${TEST_TEAM.name}`);
    });
  });

  describe('actions', () => {
    it("shouldn't allow empty name team creation", () => {
      const component = mount(
        <TeamSelectComponent {...props} />
      );

      expect(component.find('.create-team').exists()).toBe(true);
      component.find('button').simulate('click');
      expect(createTeam).not.toBeCalled();
    });

    it('should allow team creation with valid team name', () => {
      const component = mount(
        <TeamSelectComponent {...props} />
      );

      expect(component.find('.create-team').exists()).toBe(true);
      component.find('.create-team').simulate('change', { target: { name: 'teamName', value: newTeamName } });
      expect(component.find('.create-team').props().value).toEqual(newTeamName);

      component.find('button').simulate('click');
      expect(createTeam).toBeCalledWith({
        name: newTeamName
      });
    });

    it('should register a team', () => {
      const component = mount(
        <TeamSelectComponent {...propsWithTeam} />
      );

      expect(component.find('.action-text').text()).toEqual(ACTION_STATES.REGISTER);
      expect(component.find('.create-team').exists()).toBe(false);
      expect(component.find('.selected-team').text()).toEqual(TEST_TEAM.name);

      component.find('button').simulate('click');
      expect(registerTeam).toBeCalledWith({
        tournament_id: TEST_TOURNAMENT.id,
        team_id: TEST_TEAM.id,
      });
    });

    it('should be able to create even with a team', () => {
      const component = mount(
        <TeamSelectComponent {...propsWithTeam} />
      );

      // starts with selected team
      expect(component.find('.action-text').text()).toEqual(ACTION_STATES.REGISTER);
      expect(component.find('.create-team').exists()).toBe(false);
      expect(component.find('.selected-team').text()).toEqual(TEST_TEAM.name);

      // selects creating state
      component.find('.dropdown-arrow').simulate('click');
      expect(component.find('.drop-down').exists()).toBe(true);
      component.find('.create-new').simulate('click');
      expect(component.find('.drop-down').exists()).toBe(false);

      // creates new team from here
      expect(component.find('.create-team').exists()).toBe(true);
      component.find('.create-team').simulate('change', { target: { name: 'teamName', value: newTeamName } });
      expect(component.find('.create-team').props().value).toEqual(newTeamName);

      component.find('button').simulate('click');
      expect(createTeam).toBeCalledWith({
        name: newTeamName
      });
    });
  });
});
