module Authorization
  extend ActiveSupport::Concern

  class NotAuthenticated < Exception; end
  class Forbidden < Exception; end

  # User _making_ the request.
  def user_id
    return @user_id if defined?(@user_id)
    if token = request.headers['Token']
      @user_id = Token.find_by!(token: token).user_id
    end
  end

  def user_is_admin?
    !user_id.blank? && User.find(user_id).roles.include?('admin')
  end

  # Client _making_ the request.
  def client_id
    # `to_s` -> ActionController casts headers to Fixnum when they happen to be numeric. >:[
    @client_id ||= request.headers['Client-ID']&.to_s&.strip
  end

  def client
    return nil if client_id.blank?
    @client ||= Client.find_by(ident: client_id)
  end

  module ClassMethods
    # These class methods are called _outside_ the scope of an action, so render errors
    # (and thereby prevent action from starting) instead of raising an exception that
    # would bypass `BaseController.handle_exceptions`.

    def require_client!(*opts)
      prepend_before_action(*opts) do
        if client_id.blank?
          render status: 401, json: {status: 401, error: 'Missing Client-ID'}
        elsif client.nil?
          render status: 404, json: {status: 404, error: 'No such Client'}
        end
      end
    end

    def require_user!(*opts)
      prepend_before_action(*opts) do
        render_on_error { _require_user! }
      end
    end

    def require_admin!(*opts)
      before_action(*opts) do
        render_on_error do
          _require_user!
          unless user_is_admin?
            render status: 403,
              json: {status: 403, error: 'Admin privileges required'}
          end
        end
      end
    end
  end

  private

  # For the sake of sharing logic with other methods that need it executed in place
  # (as opposed to in a `before_action`).
  def _require_user!
    raise NotAuthenticated.new('Missing "Token" header') if user_id.blank?
  end

  def render_on_error
    yield
  rescue ActionController::ParameterMissing => e
    render status: 400, json: {status: 400, error: e.message}
  rescue NotAuthenticated => e
    render status: 401, json: {status: 401, error: e.message}
  end
end
