class V1::UsersController < V1::BaseController
  include Authorization

  require_admin! only: [:index]
  require_user! only: [:update, :verify_email]

  def index
    users = User.all
    render json: users, each_serializer: V1::UserPresenter, with_email: true
  end

  def show
    user = User.find(params[:id])
    with_email = user_is_admin? || user.id == user_id
    render json: user, serializer: V1::UserPresenter, with_email: with_email
  end

  def update
    unless user_is_admin? || params[:id] == user_id
      raise Forbidden.new('You do not have permission to modify this user')
    end
    user = User.find(params[:id])
    user.update_attributes!(user_params)

    # Send verification email if email_token is modified and present
    if user.valid? && user.previous_changes.keys.include?('email_token') && user.email_token.present?
      UserMailer.verify_email(user).deliver!
    end

    render json: user, serializer: V1::UserPresenter, with_email: true
  end

  def verify_email
    user = User.where(email_token: params[:email_token]).first
    raise ActiveRecord::RecordNotFound.new('Could not find a User associated with this token', User) if !user
    raise Forbidden.new('This token is associated with a different user') unless user_id == user.id
    user.verify_email!
    render json: user, serializer: V1::UserPresenter, with_email: true
  end

  private

  def user_params
    params.permit(:email, roles: [], settings: [:tos_accepted])
  end
end
