class User < ApplicationRecord
  has_many :accounts
  validates :email, allow_blank: true, uniqueness: true, format: {with: /@/}

  # Don't allow direct updates to the email column. Always go through the
  # verification process.
  # TODO allow admins to bypass this
  before_save do |user|
    # If the changes include both email and email_token being set to nil we'll
    # let it go through. If not, we revert the email change and set up
    # verification.
    if changed.include?('email') && !(changed.include?('email_token') && changes['email_token'][1].nil?)
      old_email, new_email = changes['email'].dup
      user.email = old_email
      user.email_new = new_email
      user.email_token = SecureRandom.hex(16)
    end

    user.roles.uniq!
  end

  after_commit do |user|
    if ENV['KINESIS_STREAM'].present?
      record = {
        stream_name: ENV['KINESIS_STREAM'],
        partition_key: 'tourney-passport',
        data: {
          type: 'USER_CHANGE',
          data: V1::UserPresenter.new(user, with_email: true)
        }.to_json
      }
      Rails.logger.debug("Kinesis#put_record: #{record.inspect}")
      kinesis.put_record(record) if ENV['KINESIS_STREAM'].present?
    end
  end

  def verify_email!
    update_attributes!(
      email: email_new,
      email_new: nil,
      email_token: nil
    )
  end

  def roles
    @roles = read_attribute(:roles) || []
    @roles << 'admin' if !@roles.include?('admin') && email =~ /.*@(twitch|justin)\.tv$/i
    @roles
  end

  private

  def kinesis
    @kinesis ||= Aws::Kinesis::Client.new(
      region: 'us-west-2',
      stub_responses: Rails.env.test?
    )
  end
end
