require 'rails_helper'

describe Authorization do
  controller(ApplicationController) do
    include Authorization

    require_client! only: [:requires_client]
    require_user!   only: [:requires_user]
    require_admin!  only: [:requires_admin]
    def requires_nothing; render plain: __method__; end
    def requires_client;  render plain: __method__; end
    def requires_user;    render plain: __method__; end
    def requires_admin;   render plain: __method__; end

    # RSpec does not create new instances of controllers per request within the same test;
    # `reset` is provided to reset internal state between requests.
    def reset
      remove_instance_variable(:@user_id) if defined?(@user_id)
    end

    # For easy `double()`ing
    attr_writer :user_id
  end

  def json
    JSON.parse(response.body)
  end

  before :all do
    @user0 = FactoryGirl.create(:user, id: '00000000-0000-0000-0000-000000000000')
    @user1 = FactoryGirl.create(:user, id: '11111111-1111-1111-1111-111111111111', roles: ['admin'])
    @token0 = FactoryGirl.create(:token, user: @user0)
    @token1 = FactoryGirl.create(:token, user: @user1)
  end

  describe 'headers' do
    it 'should extract User ID from "Token" header' do
      routes.draw { get 'requires_nothing' => 'anonymous#requires_nothing' }
      request.headers['Token'] = @token0.token
      get :requires_nothing
      expect(response.status).to eq(200)
      expect(controller.user_id).to eq(@user0.id)
    end
  end

  describe 'requires_nothing' do
    it 'should raise no auth/forbidden exceptions if no permissions are required' do
      routes.draw { get 'requires_nothing' => 'anonymous#requires_nothing' }
      get :requires_nothing
      expect(response.status).to eq(200)
    end
  end

  describe 'requires_client' do
    # `requires_client!` does its checks in a `before_action`,
    # so it should render an error reply instead of raising exceptions.
    it 'should reply with HTTP 401 when Client-ID header is missing' do
      routes.draw { get 'requires_client' => 'anonymous#requires_client' }
      get :requires_client
      expect(response.status).to eq(401)
      expect(json['error']).to match(/Client-ID/)

      controller.reset
      request.headers['Client-ID'] = FactoryGirl.create(:client).ident
      get :requires_client
      expect(response.status).to eq(200)
    end

    it 'should reply with HTTP 404 when Client-ID specifies a non-existent Client' do
      routes.draw { get 'requires_client' => 'anonymous#requires_client' }
      request.headers['Client-ID'] = Client.generate_ident
      get :requires_client
      expect(response.status).to eq(404)
      expect(json['error']).to match(/Client/)
    end

    it 'should not confuse the client_id param for the Client-ID header' do
      routes.draw { get 'requires_client' => 'anonymous#requires_client' }
      get :requires_client,
          params: {'client_id' => FactoryGirl.create(:client).ident}
      expect(response.status).to eq(401)
      expect(json['error']).to match(/Client-ID/)
    end
  end

  describe 'requires_user' do
    # `requires_user!` does its checks in a `before_action`,
    # so it should render an error reply instead of raising exceptions.
    it 'should reply with HTTP 401 when Token header is missing' do
      routes.draw { get 'requires_user' => 'anonymous#requires_user' }
      get :requires_user
      expect(response.status).to eq(401)
      expect(json['error']).to match(/Token/)

      controller.reset
      request.headers['Token'] = @token0.token
      get :requires_user
      expect(response.status).to eq(200)
    end

    it 'should not confuse the token param for the Token header' do
      routes.draw { get 'requires_user' => 'anonymous#requires_user' }
      get :requires_user,
        params: {'token' => @token0.token}
      expect(response.status).to eq(401)
      expect(json['error']).to match(/Token/)
    end
  end

  describe 'requires_admin' do
    # Like `requires_user!`, `requires_admin!` also does its checks in a `before_action`,
    # so it should render an error reply instead of raising exceptions.
    it "should look up a user's roles based on token" do
      routes.draw { get 'requires_admin' => 'anonymous#requires_admin' }

      get :requires_admin
      expect(response.status).to eq(401)

      # @user0 (who uses @token0) is NOT an admin.
      controller.reset
      request.headers['Token'] = @token0.token
      get :requires_admin
      expect(response.status).to eq(403)
      expect(json['error']).to match(/admin/i)

      controller.reset
      request.headers['Token'] = @token1.token
      get :requires_admin
      expect(response.status).to eq(200)
    end
  end
end
