require 'rails_helper'

RSpec.describe OmniauthCallbacksController, :type => :controller do
  include Features::Authentication

  describe '#token' do
    it "should generate/create a token if none is set" do
      expect(controller.params[:token]).to be_nil
      token = controller.send(:token)
      expect(token).to be_a(Token)
    end

    it "should prefer params token over state over creating a new token" do
      controller.params = {
        token: 'token',
        state: 'state'
      }

      token = controller.send(:token)
      expect(token.token).to eq('token')

      controller.params.delete(:token)
      controller.instance_variable_set(:@token, nil)
      token = controller.send(:token)
      expect(token.token).to eq('state')
    end
  end

  describe '#callback' do
    before :each do
      stub_oauth_login do
        request.env['omniauth.auth'] = OmniAuth.config.mock_auth[:twitch]
      end
    end

    it "should respond with a 400 if omniauth.auth is nil" do
      request.env['omniauth.auth'] = nil
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(400), response.body
      expect(request.env['omniauth.auth']).to be_nil
    end

    it "should create an account for the uid/provider if it doesn't exist" do
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:account)).to_not be_nil
      expect(assigns(:account)).to be_valid
      expect(assigns(:account)).to be_persisted
    end

    it "should create a new user for the account if none exists" do
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:user)).to be_valid
      expect(assigns(:user)).to be_persisted
    end

    it "should assign existing users to new accounts, matching the provider and uid received by oauth" do
      user = FactoryGirl.create(:user, email: 'foo@example.com')
      expect(user.verify_email!).to eq(true)
      request.env['omniauth.auth']['info']['email'] = user.email
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:user)).to eq(user)
    end

    it 'should assign new users to existing accounts, matching the email in account#email' do
      auth = Hashie::Mash.new request.env['omniauth.auth']
      account = FactoryGirl.create(:account,
        uid: auth.uid,
        provider: auth.provider,
        email: auth.info.email
      )
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:account)).to eq(account)
    end

    it 'should assign new accounts to existing users, matching the email in account#email' do
      user = FactoryGirl.create(:user)
      account = FactoryGirl.create(:account,
        provider: 'not-twitch',
        uid: 1234,
        email: request.env['omniauth.auth']['info']['email'],
        user: user
      )
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:user)).to eq(user)
    end

    it 'should re-create the user for an existing account whose user has been deleted' do
      auth = Hashie::Mash.new request.env['omniauth.auth']
      # This is actually an invalid Account, since an Account _must_ have a user, but we
      # should be robust enough to handle this case if we failed to guard against it.
      account = FactoryGirl.create(
        :account,
        uid: auth.uid,
        provider: auth.provider,
        email: auth.info.email
      )
      account.user.delete
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      expect(assigns(:user)).to_not be_nil
      account.reload
      expect(account.user).to_not be_nil
      expect(account).to be_valid
    end

    it "should render the account's user object" do
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(200), response.body
      user = assigns(:user)
      expect(user).to be_valid
      expect(json['id']).to eq(user.id)
    end

    it "should render an exception if email is invalid" do
      request.env['omniauth.auth']['info']['email'] = 'invalid email LUL'
      get :callback, params: {provider: :twitch}
      expect(response.status).to eq(422)
      expect(json['errors'].keys).to include('email')
    end

    it "should create a token for the state parameter if none exists" do
      expect(Token.where(token: 'test').all).to be_empty
      get :callback, params: {provider: :twitch, state: 'test'}
      expect(response.status).to eq(200), response.body
      expect(Token.where(token: 'test').all).to_not be_empty
    end
  end
end
