require 'rails_helper'

RSpec.describe User, :type => :model do
  describe '#roles' do
    it 'should include admin role if email is a Twitch email' do
      user = FactoryGirl.create(:user)
      expect(user.roles).to_not include('admin')
      user.email = 'foo@example.com'
      expect(user.roles).to_not include('admin')
      user.email = 'foo@twitch.tv'
      expect(user.roles).to include('admin')
      user.email = 'foo@justin.tv'
      expect(user.roles).to include('admin')
    end

    it 'should not store duplicate roles' do
      user = FactoryGirl.create(:user, roles: %w{three two one three two})
      expect(user.roles.length).to eq(3)
      expect(user.roles.sort).to eq(user.roles.uniq.sort)
    end
  end

  describe '#verify_email!' do
    it 'should not allow direct changes to email' do
      user = FactoryGirl.create(:user)
      user.update_attribute(:email, 'new@localhost')
      expect(user.email).to_not eq('new@localhost')
      expect(user.email_new).to eq('new@localhost')
      expect(user.email_token).to_not be_nil
    end

    it 'should swap email_new -> email, erase email_token' do
      user = FactoryGirl.create(:user)
      user.update_attributes(
        email_new: 'test@localhost',
        email_token: 'test'
      )
      user.verify_email!
      user.reload
      expect(user.email).to eq('test@localhost')
      expect(user.email_new).to be_nil
      expect(user.email_token).to be_nil
    end

    it 'should generate a new token if email is changed again without verifying' do
      user = FactoryGirl.create(:user)
      old_token = user.email_token
      new_email = SecureRandom.hex(8)+'@example.com'
      user.update!(email: new_email)
      expect(user.email).to be_nil
      expect(user.email_new).to eq(new_email)
      expect(user.email_token).to_not be_nil
      expect(user.email_token).to_not eq(old_token)
    end
  end

  it 'should send a message to the configured kinesis stream after_commit' do
    user = FactoryGirl.create(:user)
    expect(user.instance_eval{kinesis}).to receive(:put_record)
    user.update_attribute(:email, 'testing123@localhost')
  end

  it 'verifies uniqueness of email' do
    expect(User.create(email: 'test@example.com').verify_email!).to be(true)
    expect(User.create(email: 'test@example.com')).to be_invalid
    expect(User.where(email: 'test@example.com').count).to eq(1)
  end

  describe '#settings' do
    it 'should allow direct update to settings' do
      user = FactoryGirl.create(:user)
      user.update_attribute(:settings, { test: 1 })
      user.reload
      expect(user.settings.symbolize_keys).to eq({ test: 1 })
    end
  end
end
