require 'rails_helper'

describe V1::UserPresenter do
  before :all do
    @user = FactoryGirl.create(:user_with_account)
    @presenter = V1::UserPresenter.new(@user)
    @json = @presenter.as_json
    @presenter_with_email = V1::UserPresenter.new(@user, with_email: true)
    @json_with_email = @presenter_with_email.as_json
  end

  it 'should include roles' do
    expect(@json).to include(:roles)
  end

  it 'should only include email and email_new when given with_email option' do
    expect(@json).to_not include(:email)
    expect(@json).to_not include(:email_new)
    expect(@json_with_email).to include(:email)
    expect(@json_with_email).to include(:email_new)
  end

  it 'should include email_token outside of production' do
    expect(Rails.env.production?).to eq(false)
    expect(@presenter_with_email.as_json).to include(:email_token)
  end

  it 'should not include email_token in production' do
    Rails.env = 'production'
    expect(Rails.env.production?).to eq(true)
    expect(@presenter_with_email.as_json).to_not include(:email_token)
    Rails.env = 'test'
  end

  it 'should include previous_changes as diff' do
    @user.reload # throw away previous changes
    expect(@user.previous_changes).to eq({})
    @user.update_attributes!(email: 'test321@example.com')
    expect(@user.previous_changes).to_not eq({})
    presenter = V1::UserPresenter.new(@user)
    hash = presenter.to_hash
    expect(hash.keys).to include(:diff)
    expect(hash[:diff]).to eq(@user.previous_changes)
  end

  it 'should include diff only if previous_changes present' do
    @user.reload # throw away previous changes
    presenter = V1::UserPresenter.new(@user)
    hash = presenter.to_hash
    expect(hash.keys).to_not include(:diff)
  end

  [:name, :display_name, :avatar].each do |attr|
    it "should include #{attr} from oauth info via account if present" do
      expect(@user.accounts.first).to be_valid
      expect(@presenter.as_json).to include(attr)
      expect(@presenter.as_json[attr]).to be_present
    end
  end
end
