require 'rails_helper'

describe V1::UsersController do
  before :all do
    @nonadmin = FactoryGirl.create(:user)
    @admin = FactoryGirl.create(:user, roles: ['admin'])
    @nonadmin_token = FactoryGirl.create(:token, user: @nonadmin)
    @admin_token = FactoryGirl.create(:token, user: @admin)
  end

  describe 'GET /users' do
    it 'should only allow admins to list all users' do
      get '/v1/users'
      expect(response.status).to eq(401)
      get '/v1/users', headers: {'Token' => @nonadmin_token.token}
      expect(response.status).to eq(403), ->{pp json}
      get '/v1/users', headers: {'Token' => @admin_token.token}
      expect(response.status).to eq(200)
    end
  end

  describe 'GET /users/:user_id' do
    before :each do
      @user = FactoryGirl.create(:user)
      @token = FactoryGirl.create(:token, user: @user)
    end

    it 'should hide email for anonymous requests' do
      get "/v1/users/#{@user.id}"
      expect(response.status).to eq(200)
      ['email', 'email_new', 'email_token'].each do |attr|
        expect(json).to_not include(attr)
      end
    end

    it 'should hide email for requests from other users' do
      get "/v1/users/#{@user.id}", headers: {'Token' => @nonadmin_token.token}
      expect(response.status).to eq(200)
      ['email', 'email_new', 'email_token'].each do |attr|
        expect(json).to_not include(attr)
      end
    end

    it 'should show email for users viewing themselves' do
      get "/v1/users/#{@user.id}", headers: {'Token' => @token.token}
      expect(response.status).to eq(200)
      ['email', 'email_new', 'email_token'].each do |attr|
        expect(json).to include(attr)
      end
    end

    it 'should show email to admins' do
      get "/v1/users/#{@user.id}", headers: {'Token' => @admin_token.token}
      expect(response.status).to eq(200)
      ['email', 'email_new', 'email_token'].each do |attr|
        expect(json).to include(attr)
      end
    end
  end

  describe 'GET /verify/:email_token' do
    before :each do
      @user = FactoryGirl.create(:user)
      @token = FactoryGirl.create(:token, user: @user)
    end

    it 'should require the requestor to be logged in' do
      get '/v1/users/verify/testing'
      expect(response.status).to eq(401)
    end

    it 'should render error for non-existent email token' do
      get '/v1/users/verify/does-not-exist',
        headers: {'Token' => @token.token}
      expect(response.status).to eq(404)
      expect(json['error']).to match(/Could not find/)
    end

    it 'should only let users accept their own tokens' do
      get "/v1/users/verify/#{@user.email_token}",
        headers: {'Token' => @nonadmin_token.token}
      expect(response.status).to eq(403)
    end

    it 'should swap email and new_email on success' do
      expect(@user.email).to_not eq(@user.email_new)
      email_new = @user.email_new
      get "/v1/users/verify/#{@user.email_token}",
        headers: {'Token' => @token.token}
      expect(response.status).to eq(200)
      @user.reload
      expect(@user.email).to eq(email_new)
    end
  end

  describe 'PATCH /users/:id' do
    before :each do
      @user = FactoryGirl.create(:user)
      @token = FactoryGirl.create(:token, user: @user)
    end

    it 'should require the requestor to be logged in' do
      patch "/v1/users/#{@user.id}"
      expect(response.status).to eq(401)
    end

    it 'should not allow one user to modify another' do
      patch "/v1/users/#{@user.id}",
        headers: {'Token' => @nonadmin_token.token}
      expect(response.status).to eq(403)
    end

    it 'should allow a user to modify themselves' do
      patch "/v1/users/#{@user.id}",
        headers: {'Token' => @token.token}
      expect(response.status).to eq(200)
    end

    it 'should allow an admin to modify anyone' do
      patch "/v1/users/#{@user.id}",
        headers: {'Token' => @admin_token.token}
      expect(response.status).to eq(200)
    end

    it 'should send a verification email if email is changed' do
      expect(UserMailer).to receive_message_chain(:verify_email, :deliver!)
      patch "/v1/users/#{@user.id}",
        params: { email: 'test@localhost' },
        headers: {'Token' => @token.token}
    end

    it 'should allow updates to roles' do
      expect(@user.roles).to_not include('heavy_metal')
      patch "/v1/users/#{@user.id}",
        params: { roles: ['heavy_metal'] },
        headers: {'Token' => @admin_token.token}
      @user.reload
      expect(@user.roles).to include('heavy_metal')
    end

    it 'should allow updates to settings' do
      expect(@user.settings).to_not include('tos_accepted')
      patch "/v1/users/#{@user.id}",
        params: { settings: { tos_accepted: true } },
        headers: {'Token' => @admin_token.token}
      @user.reload
      expect(@user.settings).to include('tos_accepted')
    end
  end
end
